#include "iface.h"

#include <kernel/lemmer/core/language.h>
#include <kernel/lemmer/core/lemmer.h>
#include <kernel/lemmer/core/lemmeraux.h>
#include <kernel/lemmer/core/handler.h>
#include <kernel/lemmer/dictlib/grambitset.h>
#include <kernel/lemmer/dictlib/grammar_index.h>
#include <kernel/lemmer/tools/handler.h>
#include <kernel/lemmer/tools/order.h>
#include <library/cpp/token/nlptypes.h>
#include <library/cpp/token/token_structure.h>
#include <dict/disamb/mx_disamb/disamb.h>
#include <library/cpp/langs/langs.h>
#include <util/charset/wide.h>
#include <string.h>

#if defined(_win_)
    #define EXPORT __declspec(dllexport)
#else
    #define EXPORT
#endif

using NLemmer::TLemmerHandler;
using NLemmer::TLemmerResult;
using NLemmer::TLemmerResults;
using NLemmer::ILemmerResultsHandler;
using NLemmer::TAnalyzeWordOpt;
using NLemmer::TLemmerResultOrderGreater;

// Plain form of lemmer output
struct TDisambLemmerResult {
    TString Form;
    TString Lemma;
    TString LexicalFeatures;
    TString InflectionalFeatures;
    float Weight;

    TDisambLemmerResult(const TString& frm, const TString& lem,
                        const TString& lexf, const TString& inflf,
                        float w)
        : Form(frm)
        , Lemma(lem)
        , LexicalFeatures(lexf)
        , InflectionalFeatures(inflf)
        , Weight (w) {}
};

// Produce a vector of plain form results from original lemmer structures
static void FillCallbackData(TVector<TDisambLemmerResult>& cbdata, const TLemmerResults* resptr) {
    if (!resptr)
        return;

    TLemmerResults res = *resptr;
    StableSort(res.begin(), res.end(), TLemmerResultOrderGreater());

    for (TLemmerResults::const_iterator it = res.begin(); it != res.end(); ++it) {
        const TYandexLemma& analysis = it->first;
        float weight = it->second;
        TString lemma = WideToUTF8(analysis.GetText(), analysis.GetTextLength());
        TString form = WideToUTF8(analysis.GetNormalizedForm(), analysis.GetNormalizedFormLength());
        TString lexfeatures = sprint_grammar(analysis.GetStemGram());

        size_t homcount = analysis.FlexGramNum(); // number of homonyms
        if (!homcount) {
            cbdata.push_back(TDisambLemmerResult(form, lemma, lexfeatures, TString(), weight));
        } else {
            for (size_t i = 0; i < homcount; ++i) {
                TString inflfeatures = sprint_grammar(analysis.GetFlexGram()[i]);
                cbdata.push_back(TDisambLemmerResult(form, lemma, lexfeatures, inflfeatures, weight));
            }
        }
    }
}

class TLemmerCallbackHost : public ILemmerResultsHandler {
private:
    DL_ResultCallback ResultCallback;

public:
    TLemmerCallbackHost(DL_ResultCallback cb)
        : ResultCallback(cb) {}

    bool OnLemmerResults(const TWideToken& tok, NLP_TYPE type, const TLemmerResults* results) override {
        TString text(WideToUTF8(tok.Token, tok.Leng));
        if (type == NLP_WORD) {
            TVector<TDisambLemmerResult> cbdata;
            FillCallbackData(cbdata, results);
            (* ResultCallback)(text.c_str(), (DL_ResultArrayHandle)&cbdata);
        }
        return true;
    }

    void Flush() override {}
};

class TDisambLemmerSession {
public:
    TDisambLemmerSession(DL_ResultCallback cb, ELanguage lang)
        : CallbackHost(cb)
        , Parameters(TLangMask(lang), nullptr, TAnalyzeWordOpt::DefaultLemmerTestOpt())
        , DisambHandler(CallbackHost)
        , LemmerHandler(Parameters, DisambHandler) { }

    void NextWord(const char * text) {
        TUtf16String widetext(UTF8ToWide(text, strlen(text)));
        TWideToken token(widetext.c_str(), widetext.length());
        LemmerHandler.OnToken(token, token.Leng, NLP_WORD);
    }

    void Flush() {
        LemmerHandler.Flush();
    }

private:
    TLemmerCallbackHost CallbackHost;
    TLemmerHandler::TParams Parameters;
    NMxNetDisamb::TDisambModel DisambHandler;
    TLemmerHandler LemmerHandler;
};

extern "C" {
    // Session
    EXPORT DL_SessionHandle DL_CreateSession(DL_ResultCallback cb, const char* lang) {
        return (DL_SessionHandle) new TDisambLemmerSession(cb, LanguageByName(lang));
    }

    EXPORT void DL_DestroySession(DL_SessionHandle proc) {
        delete (TDisambLemmerSession *) proc;
    }

    EXPORT void DL_NextWord(DL_SessionHandle proc, const char * text) {
        ((TDisambLemmerSession *) proc)->NextWord(text);
    }

    EXPORT void DL_Flush(DL_SessionHandle proc) {
        ((TDisambLemmerSession *) proc)->Flush();
    }

    // Result array
    EXPORT int DL_GetResultCount(DL_ResultArrayHandle handle) {
        const TVector<TDisambLemmerResult> * resarr = (const TVector<TDisambLemmerResult> *)handle;
        return (int)(resarr->size());
    }

    EXPORT DL_ResultHandle DL_GetResultAt(DL_ResultArrayHandle handle, int idx) {
        const TVector<TDisambLemmerResult> * resarr = (const TVector<TDisambLemmerResult> *)handle;
        return (DL_ResultHandle) &(resarr->at(idx));
    }

    // Individual result
    EXPORT const char * DL_GetWordForm(DL_ResultHandle handle) {
        const TDisambLemmerResult * res = (const TDisambLemmerResult *) handle;
        return res->Form.c_str();
    }

    EXPORT const char * DL_GetLemma(DL_ResultHandle handle) {
        const TDisambLemmerResult * res = (const TDisambLemmerResult *) handle;
        return res->Lemma.c_str();
    }

    EXPORT const char * DL_GetLexicalFeatures(DL_ResultHandle handle) {
        const TDisambLemmerResult * res = (const TDisambLemmerResult *) handle;
        return res->LexicalFeatures.c_str();
    }

    EXPORT const char * DL_GetInflectionFeatures(DL_ResultHandle handle) {
        const TDisambLemmerResult * res = (const TDisambLemmerResult *) handle;
        return res->InflectionalFeatures.c_str();
    }

    EXPORT float DL_GetWeight(DL_ResultHandle handle) {
        const TDisambLemmerResult * res = (const TDisambLemmerResult *) handle;
        return res->Weight;
    }
} // extern "C"
