#include "iface.h"

#include <kernel/lemmer/core/language.h>
#include <kernel/lemmer/core/lemmer.h>
#include <kernel/lemmer/core/lemmeraux.h>
#include <kernel/lemmer/core/handler.h>
#include <kernel/lemmer/dictlib/grambitset.h>
#include <kernel/lemmer/dictlib/grammar_index.h>
#include <kernel/lemmer/tools/handler.h>
#include <kernel/lemmer/tools/order.h>
#include <library/cpp/token/nlptypes.h>
#include <library/cpp/token/token_structure.h>
#include <dict/disamb/mx_disamb/disamb.h>
#include <library/cpp/langs/langs.h>
#include <util/charset/wide.h>
#include <string.h>

#if defined(_win_)
    #define EXPORT __declspec(dllexport)
#else
    #define EXPORT
#endif

using NLemmer::TLemmerHandler;
using NLemmer::TLemmerResult;
using NLemmer::TLemmerResults;
using NLemmer::ILemmerResultsHandler;
using NLemmer::TAnalyzeWordOpt;
using NLemmer::TLemmerResultOrderGreater;

// Plain form of lemmer output
struct TDisambLemmerResult {
    TString Form;
    TString Lemma;
    TString LexicalFeatures;
    TString InflectionalFeatures;
    float Weight;

    TDisambLemmerResult(const TString& frm, const TString& lem,
                        const TString& lexf, const TString& inflf,
                        float w)
        : Form(frm)
        , Lemma(lem)
        , LexicalFeatures(lexf)
        , InflectionalFeatures(inflf)
        , Weight (w) {}
};

// Produce a vector of plain form results from original lemmer structures
static void FillLemmerResults(TVector<TDisambLemmerResult>& cbdata, const TLemmerResults* resptr) {
    if (!resptr)
        return;

    TLemmerResults res = *resptr;
    StableSort(res.begin(), res.end(), TLemmerResultOrderGreater());

    for (TLemmerResults::const_iterator it = res.begin(); it != res.end(); ++it) {
        const TYandexLemma& analysis = it->first;
        float weight = it->second;
        TString lemma = WideToUTF8(analysis.GetText(), analysis.GetTextLength());
        TString form = WideToUTF8(analysis.GetNormalizedForm(), analysis.GetNormalizedFormLength());
        TString lexfeatures = sprint_grammar(analysis.GetStemGram());

        size_t homcount = analysis.FlexGramNum(); // number of homonyms
        if (!homcount) {
            cbdata.push_back(TDisambLemmerResult(form, lemma, lexfeatures, TString(), weight));
        } else {
            for (size_t i = 0; i < homcount; ++i) {
                TString inflfeatures = sprint_grammar(analysis.GetFlexGram()[i]);
                cbdata.push_back(TDisambLemmerResult(form, lemma, lexfeatures, inflfeatures, weight));
            }
        }
    }
}

struct TWordResults {
    TString Text;
    TVector<TDisambLemmerResult> Analyses;

    TWordResults() {}
    TWordResults(const TWordResults& other)
        : Text (other.Text)
        , Analyses (other.Analyses) {}

    TWordResults(const TString& txt, const TLemmerResults * results)
        : Text(txt)  {
        FillLemmerResults(Analyses, results);
    }
};

typedef TVector<TWordResults> TPhraseResults;

static void WordResultsConcatenator(DL_CallbackArgument cbarg, DL_WordResults wr) {
    TPhraseResults * phrase = (TPhraseResults *) cbarg;
    const TWordResults * word = (const TWordResults *) wr;
    phrase->push_back(*word);
}

class TLemmerCallbackHost : public ILemmerResultsHandler {
private:
    DL_ResultCallback ResultCallback;
    DL_CallbackArgument CallbackArgument;

public:
    TLemmerCallbackHost(DL_ResultCallback cb, DL_CallbackArgument cbarg)
        : ResultCallback(cb)
        , CallbackArgument(cbarg) {}

    void SetCallback(DL_ResultCallback cb, DL_CallbackArgument cbarg) {
        ResultCallback = cb;
        CallbackArgument = cbarg;
    }

    bool OnLemmerResults(const TWideToken& tok, NLP_TYPE type, const TLemmerResults* results) override {
        if (type == NLP_WORD) {
            TWordResults wr(WideToUTF8(tok.Token, tok.Leng), results);
            if (ResultCallback)
                (* ResultCallback)(CallbackArgument, (DL_WordResults)&wr);
        }
        return true;
    }

    void Flush() override {}
};

class TDisambLemmerSession {
public:
    TDisambLemmerSession(DL_ResultCallback cb, DL_CallbackArgument cbarg, ELanguage lang)
        : CallbackHost(cb, cbarg)
        , Parameters(TLangMask(lang), nullptr, TAnalyzeWordOpt::DefaultLemmerTestOpt())
        , DisambHandler(CallbackHost)
        , LemmerHandler(Parameters, DisambHandler) { }

    void NextWord(const char * text) {
        TUtf16String widetext(UTF8ToWide(text, strlen(text)));
        TWideToken token(widetext.c_str(), widetext.length());
        LemmerHandler.OnToken(token, token.Leng, NLP_WORD);
    }

    void SetCallback(DL_ResultCallback cb, DL_CallbackArgument cbarg) {
        CallbackHost.SetCallback(cb, cbarg);
    }

    void Flush() {
        LemmerHandler.Flush();
    }

    ~TDisambLemmerSession() {
        Flush();
    }

private:
    TLemmerCallbackHost CallbackHost;
    TLemmerHandler::TParams Parameters;
    NMxNetDisamb::TDisambModel DisambHandler;
    TLemmerHandler LemmerHandler;
};

extern "C" {
    // Static initialization
    EXPORT int DL_Precharge(const char* lang) {
        const char* phrase[] = { "a", "\xd0\xb0", nullptr };
        DL_PhraseResults res = DL_AnalyzePhrase(phrase, lang);
        int exitcode = (DL_GetWordCount(res) > 0);
        DL_DestroyPhraseResults(res);
        return exitcode;
    }

    // Session
    EXPORT DL_SessionHandle DL_CreateSession(DL_ResultCallback cb, DL_CallbackArgument cbarg, const char* lang) {
        return (DL_SessionHandle) new TDisambLemmerSession(cb, cbarg, LanguageByName(lang));
    }

    EXPORT void DL_SetCallback(DL_SessionHandle proc, DL_ResultCallback cb, DL_CallbackArgument cbarg) {
        ((TDisambLemmerSession *) proc)->SetCallback(cb, cbarg);
    }

    EXPORT void DL_DestroySession(DL_SessionHandle proc) {
        delete (TDisambLemmerSession *) proc;
    }

    EXPORT void DL_NextWord(DL_SessionHandle proc, const char * text) {
        ((TDisambLemmerSession *) proc)->NextWord(text);
    }

    EXPORT void DL_Flush(DL_SessionHandle proc) {
        ((TDisambLemmerSession *) proc)->Flush();
    }

    // Phrase results
    EXPORT DL_PhraseResults DL_AnalyzePhrase(const char ** words, const char* lang) {
        TPhraseResults * res = new TPhraseResults;
        TDisambLemmerSession session((DL_ResultCallback) WordResultsConcatenator,
                                     (DL_CallbackArgument) res,
                                      LanguageByName(lang) );
        for (const char ** pword = words; *pword; ++pword) {
            session.NextWord(*pword);
        }
        session.Flush();
        return (DL_PhraseResults) res;
    }

    EXPORT int DL_GetWordCount(DL_PhraseResults handle) {
        const TPhraseResults * res = (const TPhraseResults *)handle;
        return (int)(res->size());
    }

    EXPORT DL_WordResults DL_GetWordResultsAt(DL_PhraseResults handle, int idx) {
        const TPhraseResults * res = (const TPhraseResults *)handle;
        return (DL_WordResults) &(res->at(idx));
    }

    EXPORT void DL_DestroyPhraseResults(DL_PhraseResults handle) {
        delete (const TPhraseResults *)handle;
    }

    // Word results
    EXPORT const char * DL_GetWordText(DL_WordResults handle) {
        const TWordResults * res = (const TWordResults *)handle;
        return res->Text.c_str();
    }

    EXPORT int DL_GetAnalysisCount(DL_WordResults handle) {
        const TWordResults * res = (const TWordResults *)handle;
        return (int)(res->Analyses.size());
    }

    EXPORT DL_Analysis DL_GetAnalysisAt(DL_WordResults handle, int idx) {
        const TWordResults * res = (const TWordResults *)handle;
        return (DL_Analysis) &(res->Analyses.at(idx));
    }

    // Single analysis
    EXPORT const char * DL_GetWordForm(DL_Analysis handle) {
        const TDisambLemmerResult * res = (const TDisambLemmerResult *) handle;
        return res->Form.c_str();
    }

    EXPORT const char * DL_GetLemma(DL_Analysis handle) {
        const TDisambLemmerResult * res = (const TDisambLemmerResult *) handle;
        return res->Lemma.c_str();
    }

    EXPORT const char * DL_GetLexicalFeatures(DL_Analysis handle) {
        const TDisambLemmerResult * res = (const TDisambLemmerResult *) handle;
        return res->LexicalFeatures.c_str();
    }

    EXPORT const char * DL_GetInflectionFeatures(DL_Analysis handle) {
        const TDisambLemmerResult * res = (const TDisambLemmerResult *) handle;
        return res->InflectionalFeatures.c_str();
    }

    EXPORT float DL_GetWeight(DL_Analysis handle) {
        const TDisambLemmerResult * res = (const TDisambLemmerResult *) handle;
        return res->Weight;
    }
} // extern "C"
