#pragma once

#include <stddef.h>

#if defined(__cplusplus)
extern "C" {
#endif

// A helper function to trigger loading of all static objects in morphology.
// No need to use it unless you fork many processes from a single parent.
int DL_Precharge(const char* lang);

// A single analysis includes: normalized word form, lemma, lexical and
// inflectional features. It also has a weight (can be zero if the underlying
// engine does not support it); parses are sorted by decreasing weight within
// the result array, so the most probable always comes first.
typedef void * DL_Analysis;
const char * DL_GetWordForm(DL_Analysis res);
const char * DL_GetLemma(DL_Analysis res);
const char * DL_GetLexicalFeatures(DL_Analysis res);
const char * DL_GetInflectionFeatures(DL_Analysis res);
float DL_GetWeight(DL_Analysis res);

// The word result array keeps all alternative analyses of a single word.
// Besides the word itself, it has a length and an element accessor
typedef void * DL_WordResults;

const char * DL_GetWordText(DL_WordResults wordres);
int DL_GetAnalysisCount(DL_WordResults wordres);
DL_Analysis DL_GetAnalysisAt(DL_WordResults wordres, int idx);

// A Session stores local context. Words are fed one by one through NextWord,
// and the result is communicated asynchronously via ResultCallback.
// There is a Flush() function to flush the inner state and clear the context.
// (This implies an external tokenizer, both by words and by sentences.)
// NOTE: besides the Session itself, all other method arguments
// are short-lived. They are guaranteed to persist only within the
// callback that receives them in arguments. Don't store them.
typedef void * DL_CallbackArgument;
typedef void (* DL_ResultCallback)(DL_CallbackArgument, DL_WordResults);
typedef void * DL_SessionHandle;

DL_SessionHandle DL_CreateSession(DL_ResultCallback cb, DL_CallbackArgument cbarg, const char* language);
void DL_SetCallback(DL_SessionHandle proc, DL_ResultCallback cb, DL_CallbackArgument cbarg);
void DL_NextWord(DL_SessionHandle proc, const char * word);
void DL_Flush(DL_SessionHandle proc);
void DL_DestroySession(DL_SessionHandle proc);

// Alternatively, one can supply an entire sentence and get a bunch of results back,
// without the callback hassle. The resulting object must be disposed of properly.
typedef void * DL_PhraseResults;
DL_PhraseResults DL_AnalyzePhrase(const char ** words, const char* language);
int DL_GetWordCount(DL_PhraseResults phraseres);
DL_WordResults DL_GetWordResultsAt(DL_PhraseResults phraseres, int idx);
void DL_DestroyPhraseResults(DL_PhraseResults);

#if defined(__cplusplus)
} // extern "C"
#endif
