package ru.yandex.apphost;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;

/**
 * @author imakunin
 */
public final class AppHost {

  private final AtomicBoolean started = new AtomicBoolean(true);
  private volatile long impl;

  public static AppHost start(int port, int threadsCount, Consumer<AppHostRequest> callback) {
    final AppHost instance = new AppHost();
    instance.start0(port, threadsCount, callback);
    return instance;
  }

  private AppHost() {
  }

  public void stop() {
    if (started.compareAndSet(true, false)) {
      stop0();
    }
  }

  private native void start0(int port, int threadsCount, Consumer<AppHostRequest> callback);

  private native void stop0();

  private static native void registerNatives();

  private static void loadLibrary(String name) throws IOException {
    try {
      System.loadLibrary(name);
      return;
    } catch (UnsatisfiedLinkError ignored) {
    }

    final String libFileName = System.mapLibraryName(name);
    try (InputStream in = AppHost.class.getClassLoader().getResourceAsStream(libFileName)) {
      if (in == null) {
        throw new IOException("Missing JNI library - " + libFileName);
      }
      final int pos = libFileName.lastIndexOf('.');
      final File file = File.createTempFile(libFileName.substring(0, pos), libFileName.substring(pos));
      file.deleteOnExit();
      Files.copy(in, file.toPath(), StandardCopyOption.REPLACE_EXISTING);
      tmpFilename = file.getAbsolutePath();
    }

    System.load(tmpFilename);
  }

  @Override
  protected void finalize() throws Throwable {
    if (tmpFilename != null) {
      Files.deleteIfExists(Paths.get(tmpFilename));
    }
    super.finalize();
  }

  private static volatile String tmpFilename = null;

  static {
    try {
      loadLibrary("apphost");
    } catch (IOException e) {
      throw new UnsatisfiedLinkError(e.getMessage());
    }
    registerNatives();
  }

}
