#include <util/generic/string.h>
#include <util/charset/wide.h>

#include <kernel/lemmer/core/langcontext.h>
#include <kernel/qtree/richrequest/richnode.h>

#include <kernel/snippets/passages/passages.h>
#include <kernel/snippets/smartcut/smartcut.h>
#include <kernel/snippets/strhl/goodwrds.h>
#include <kernel/snippets/titles/make_title/make_title.h>

#include "ru_yandex_common_util_highlight_Highlighter.h"

static jfieldID hiliteFID;
static jfieldID refcntFID;
static jmethodID refMID;
static jmethodID unrefMID;
static jclass iaeClass;
static jclass eeeClass;

//static TAtomicCounter _cnt;

static inline TInlineHighlighter* getHiliter(JNIEnv *env, jobject obj) {
    return (TInlineHighlighter*) env->GetLongField(obj, hiliteFID);
}

static inline void setHiliter(JNIEnv *env, jobject obj, TInlineHighlighter *ptr) {
    env->SetLongField(obj, hiliteFID, (long) ptr);
}

static inline int ref(JNIEnv *env, jobject obj) {
    return env->CallIntMethod(obj, refMID);
}

static inline void unref(JNIEnv *env, jobject obj) {
    int cnt = env->CallIntMethod(obj, unrefMID);
    if (cnt <= 0) {
        TInlineHighlighter* hiliter = getHiliter(env, obj);
        if (hiliter) {
            setHiliter(env, obj, nullptr);
            delete hiliter;
//            _cnt.Dec();
//            Cout << "--------> " << _cnt.Val() << Endl;
        }
    }
}

struct Ref {
    JNIEnv *env;
    jobject obj;
    bool ok;
    Ref(JNIEnv *env, jobject obj) : env(env), obj(obj) {
        ok = ref(env, obj);
     }
    ~Ref() {
        if (ok) unref(env, obj);
     }
};

jint JNI_OnLoad(JavaVM *jvm, void *)
{
    JNIEnv *env;
    if (jvm->GetEnv((void**)&env, JNI_VERSION_1_6) != JNI_OK) {
        return -1;
    }

    iaeClass = (jclass)env->NewGlobalRef(env->FindClass("java/lang/IllegalArgumentException"));
    eeeClass = (jclass)env->NewGlobalRef(env->FindClass("java/lang/Error"));

    return JNI_VERSION_1_6;
}

JNIEXPORT void JNICALL Java_ru_yandex_common_util_highlight_Highlighter_registerNatives(JNIEnv *env, jclass objClass)
{
    hiliteFID = env->GetFieldID(objClass, "implPointer", "J");
    refcntFID = env->GetFieldID(objClass, "refCnt", "I");
    refMID = env->GetMethodID(objClass, "ref", "()Z");
    unrefMID = env->GetMethodID(objClass, "unref", "()I");
}

JNIEXPORT void JNICALL Java_ru_yandex_common_util_highlight_Highlighter_destroy(JNIEnv *env, jobject obj)
{
    unref(env, obj);
}

void initHiliter(JNIEnv *env, jobject obj, TRichTreePtr tree)
{
    TInlineHighlighter* hiliter = new TInlineHighlighter();
    if (hiliter) {
//        _cnt.Inc();
        setHiliter(env, obj, hiliter);
        env->SetIntField(obj, refcntFID, 1);
    } else {
        env->ThrowNew(eeeClass, "Could not create TInlineHighlighter");
        return;
    }

    try {
        hiliter->AddRequest(*tree->Root);
    } catch (const yexception& e) {
        env->ThrowNew(iaeClass, e.what());
    }
}

JNIEXPORT void JNICALL Java_ru_yandex_common_util_highlight_Highlighter_initByTree(JNIEnv *env, jobject obj, jstring qtreeString)
{
    const char* qtree = env->GetStringUTFChars(qtreeString, nullptr);

    try {
        TRichTreePtr tree = DeserializeRichTree(DecodeRichTreeBase64(qtree));
        initHiliter(env, obj, tree);
    } catch (const yexception& e) {
      env->ThrowNew(iaeClass, e.what());
    }

    env->ReleaseStringUTFChars(qtreeString, qtree);
}

JNIEXPORT void JNICALL Java_ru_yandex_common_util_highlight_Highlighter_initByText(JNIEnv *env, jobject obj, jstring reqString)
{
    const char* req = env->GetStringUTFChars(reqString, nullptr);

    TUtf16String reqWtroka = UTF8ToWide(req);

    try {
        TRichTreePtr tree = CreateRichTree(reqWtroka, TCreateTreeOptions(TLanguageContext(LI_DEFAULT_REQUEST_LANGUAGES)));
        initHiliter(env, obj, tree);
    } catch (const yexception& e) {
        env->ThrowNew(iaeClass, e.what());
    }

    env->ReleaseStringUTFChars(reqString, req);
}

JNIEXPORT jstring JNICALL Java_ru_yandex_common_util_highlight_Highlighter_highlight__Ljava_lang_String_2(JNIEnv *env, jobject obj, jstring textString)
{
    Ref r(env, obj);
    if (!r.ok) {
        return textString;
    }

    const char* text = env->GetStringUTFChars(textString, nullptr);

    TUtf16String wtext = UTF8ToWide(text);
    TInlineHighlighter* hiliter = getHiliter(env, obj);
    hiliter->PaintPassages(wtext);

    env->ReleaseStringUTFChars(textString, text);

    return env->NewStringUTF(WideToUTF8(wtext).data());
}

JNIEXPORT jstring JNICALL Java_ru_yandex_common_util_highlight_Highlighter_highlight__Ljava_lang_String_2Ljava_lang_String_2Z(JNIEnv *env, jclass, jstring jtext, jstring jquery, jboolean isTree)
{
    try {
        if (!jtext) {
            env->ThrowNew(iaeClass, "text is null");
            return nullptr;
        }
        if (!jquery) {
            env->ThrowNew(iaeClass, "query is null");
            return nullptr;
        }
        const char* ctext = env->GetStringUTFChars(jtext, nullptr);
        if (!ctext) {
            return nullptr;
        }
        TUtf16String wtext = UTF8ToWide(ctext);

        const char* cquery = env->GetStringUTFChars(jquery, nullptr);
        if (!cquery) {
            env->ReleaseStringUTFChars(jtext, ctext);
            return nullptr;
        }

        TRichTreePtr tree = nullptr;
        if (isTree) {
            tree = DeserializeRichTree(DecodeRichTreeBase64(cquery));
        } else {
            tree = CreateRichTree(UTF8ToWide(cquery), TCreateTreeOptions(TLanguageContext(LI_DEFAULT_REQUEST_LANGUAGES)));
        }

        TInlineHighlighter ih;
        ih.AddRequest(*tree->Root);
        ih.PaintPassages(wtext);
        env->ReleaseStringUTFChars(jtext, ctext);
        env->ReleaseStringUTFChars(jquery, cquery);
        return env->NewStringUTF(WideToUTF8(wtext).data());
    } catch (const yexception& e) {
        env->ThrowNew(iaeClass, e.what());
    } catch (...) {
        env->ThrowNew (eeeClass, "Unknown error");
    }
    return nullptr;
}

JNIEXPORT jstring JNICALL Java_ru_yandex_common_util_highlight_Highlighter_cut(JNIEnv *env, jclass, jstring jtext, jlong len)
{
    try {
        if (!jtext) {
            env->ThrowNew(iaeClass, "text is null");
            return nullptr;
        }
        const char* ctext = env->GetStringUTFChars(jtext, nullptr);
        if (!ctext) {
            return nullptr;
        }

        TUtf16String wtext = UTF8ToWide(ctext);
        NSnippets::TTextCuttingOptions options;
        options.AddEllipsisToShortText = true;
        wtext = NSnippets::SmartCutSymbol(wtext, len, options);

        env->ReleaseStringUTFChars(jtext, ctext);
        return env->NewStringUTF(WideToUTF8(wtext).data());
    } catch (const yexception& e) {
        env->ThrowNew(iaeClass, e.what());
    } catch (...) {
        env->ThrowNew (eeeClass, "Unknown error");
    }
    return nullptr;
}

JNIEXPORT jstring JNICALL Java_ru_yandex_common_util_highlight_Highlighter_makeTitle(JNIEnv *env, jclass, jstring jtext, jlong len, jfloat size)
{
    try {
        if (!jtext) {
            env->ThrowNew(iaeClass, "text is null");
            return nullptr;
        }
        const char* ctext = env->GetStringUTFChars(jtext, nullptr);
        if (!ctext) {
            return nullptr;
        }
        TUtf16String wtext = UTF8ToWide(ctext);
        TUtf16String title = NSnippets::MakeTitleString(wtext, nullptr, len, size);
        env->ReleaseStringUTFChars(jtext, ctext);
        return env->NewStringUTF(WideToUTF8(title).data());
    } catch (const yexception& e) {
        env->ThrowNew(iaeClass, e.what());
    } catch (...) {
        env->ThrowNew (eeeClass, "Unknown error");
    }
    return nullptr;
}

JNIEXPORT jstring JNICALL Java_ru_yandex_common_util_highlight_Highlighter_makeQTree(JNIEnv *env, jclass, jstring jtext)
{
    try {
        if (!jtext) {
            env->ThrowNew(iaeClass, "query is null");
            return nullptr;
        }
        const char* ctext = env->GetStringUTFChars(jtext, nullptr);
        if (!ctext) {
            return nullptr;
        }
        TString qtree = CreatePackedTree(UTF8ToWide(ctext), TCreateTreeOptions(), nullptr);
        env->ReleaseStringUTFChars(jtext, ctext);
        return env->NewStringUTF(qtree.data());
    } catch (const yexception& e) {
        env->ThrowNew(iaeClass, e.what());
    } catch (...) {
        env->ThrowNew (eeeClass, "Unknown error");
    }
    return nullptr;
}
