package ru.yandex.common.util.highlight;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;

/**
 * User: lyadzhin
 * Date: 02.09.13 13:23
 */
public final class Highlighter {
  public static final String START_MARKER = "\u0007[";
  public static final String END_MARKER = "\u0007]";

  static {
    try {
      loadLibrary("highlight");
    } catch (IOException e) {
      throw new UnsatisfiedLinkError(e.getMessage());
    }
    registerNatives();
  }

  public Highlighter() {
  }

  private static void loadLibrary(final String name) throws IOException {
    try {
      System.loadLibrary(name);
      return;
    } catch (UnsatisfiedLinkError e) {
      // Ignore the error and try to extract the JNI module from .jar
    }

    final String libFileName = System.mapLibraryName(name);

    final File file;
    try (InputStream in = Highlighter.class.getClassLoader().getResourceAsStream(libFileName)) {
      if (in == null) {
        throw new IOException("Missing JNI library - " + libFileName);
      }

      final int pos = libFileName.lastIndexOf('.');
      file = File.createTempFile(libFileName.substring(0, pos), libFileName.substring(pos));
      file.deleteOnExit();
      Files.copy(in, file.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    System.load(file.getAbsolutePath());
  }

  public static Highlighter createByText(final String text) {
    if (text == null) {
      throw new NullPointerException("text");
    }
    final Highlighter highlighter = new Highlighter();
    highlighter.initByText(text);
    return highlighter;
  }

  public static Highlighter createByTree(final String encodedQueryTree) {
    if (encodedQueryTree == null) {
      throw new NullPointerException("encodedQueryTree");
    }
    final Highlighter highlighter = new Highlighter();
    highlighter.initByTree(encodedQueryTree);
    return highlighter;
  }


  private static native void registerNatives();

  public static native String highlight(String text, String query, boolean isTree) throws IllegalArgumentException;

  public static native String cut(String text, long maxLen) throws IllegalArgumentException;

  public static native String makeTitle(String text, long maxLen, float fontSize) throws IllegalArgumentException;

  public static native String makeQTree(String query) throws IllegalArgumentException;


  public native String highlight(String text);

  public native void destroy();


  private native void initByText(String query);

  private native void initByTree(String queryTree);

  long getImplPointer() {
    return implPointer;
  }

  @Override
  protected void finalize() throws Throwable {
    destroy();
    super.finalize();
  }

  private volatile long implPointer;
  private volatile int refCnt;

  private static final AtomicIntegerFieldUpdater<Highlighter> refCntUpdater =
      AtomicIntegerFieldUpdater.newUpdater(Highlighter.class, "refCnt");

  boolean ref() {
    do {
      final int cnt = refCnt;
      if (cnt <= 0) {
        return false;
      }
      if (refCntUpdater.compareAndSet(this, cnt, cnt + 1)) {
        return true;
      }
    } while (true);
  }

  int unref() {
    return refCntUpdater.decrementAndGet(this);
  }
}
