package ru.yandex.common.util.highlight;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;

/**
 * @author zador
 * @date 2015-05-09 20:07
 */
public class HighlighterTest {
  // путина
  private static final String tree = "cHiclVPPa9NQHP9-X7vu-agQNgozMuh6aS1Mg3ooHlSKhwke6rxIQJkVJB6UER2EnopFqBNEEEFEGCgVT6EIwgS3inhxKCb6B4gnRfDq2e-LL2mS1sJy-ib5fn58P-99xUWR56DBHC9ChRkwk_cGfse_5W16W15fhxJU4TAcy_Mc9QD1gAEnYQkacB5Wcpb7m99HeIzwFBPAlwhbCPRUH2ZFXUlkiiyQgBgtjNJaGFJCRKO_z4gLI043Ek5x4f9WrV9_9g6NbowxakAN618YRw31xCglqKCBS-tt1rhMM5M9-6gQww6NzQVSNVjOdp-8_WCy567JXrhNoYHOvD4ZmbXAQdsQCV7C4RA3IFzPbXLCQIQ4NAHhDnZM7KUAB1MAHgFYdzAiMGVXU_3ttpTAAAGxXnSEvRjv9TsyY8WNd0ygaTE2LW9xjnyKcy5W0JpenW4jUP4y5GFV35Zxw0jcdKkocNZ4xlTelUTeUji0GM-YO2C3dpNxM0-es96295reMsTFdOZ3VE3T-HdVLTXexOrNWP9tWQfhX9vFaam4XgVEaRtS-t4kS1KuRdmZ7xjnhW_LV07pfSbOpfYjt7p28-r1SzoeKGFlwmo8-uHvD3dDYZJbsYPBMal_ah-w8SA8niNC_dIwuj2s66YHpuTKNwL76DC7PAaU6fZcM3nvmH16TCN2UycZY2_uodAwLEHHcqhJkbHhXSxlZVoVUF_WcUEEX2b2cdQLnBU-fTxzfF6GUITFf1mkezKF72edE_M_v36uRT3E_Bci3lOa";

  @Rule
  public ExpectedException exception = ExpectedException.none();

  @Test
  public void testHighlightNull1() throws Exception {
    exception.expect(IllegalArgumentException.class);
    Highlighter.highlight(null, tree, true);
  }

  @Test
  public void testHighlightNull2() throws Exception {
    exception.expect(IllegalArgumentException.class);
    Highlighter.highlight("путина", null, true);
  }

  @Test
  public void testHighlightNull3() throws Exception {
    exception.expect(IllegalArgumentException.class);
    Highlighter.highlight(null, "путина", false);
  }

  @Test
  public void testHighlightNull4() throws Exception {
    exception.expect(IllegalArgumentException.class);
    Highlighter.highlight("путина", null, false);
  }

  @Test
  public void testHighlightBadTree1() throws Exception {
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("can not load pod type");
    Highlighter.highlight("", "", true);
  }

  @Test
  public void testHighlightBadTree2() throws Exception {
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("incorrect input length for base64 decode");
    Highlighter.highlight("", tree.substring(1), true);
  }

  @Test
  public void testHighlightBadTree3() throws Exception {
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("Unsupported compression format");
    Highlighter.highlight("", tree.substring(tree.length() / 4), true);
  }

  @Test
  public void testHighlightBadTree4() throws Exception {
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("inflate error(invalid literal/lengths set");
    Highlighter.highlight("", tree.replace('-', '_'), true);
  }

  @Test
  public void testHighlightBadQuery1() throws Exception {
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("Search request is empty");
    assertEquals("", Highlighter.highlight("", " ", false));
  }

  @Test
  public void testHighlightBadQuery2() throws Exception {
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("Search request is empty");
    assertEquals("", Highlighter.highlight("", "!", false));
  }

  @Test
  public void testHighlightBadQuery3() throws Exception {
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("Search request is empty");
    assertEquals("", Highlighter.highlight("", "...", false));
  }

  @Test
  public void testHighlightBadQuery4() throws Exception {
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("Search request is empty");
    assertEquals("", Highlighter.highlight("", "?,: =+...", false));
  }

  @Test
  public void testHighlightEmpty() throws Exception {
    assertEquals("", Highlighter.highlight("", tree, true));
    assertEquals(" ", Highlighter.highlight(" ", tree, true));
    assertEquals("", Highlighter.highlight("", "", false));
    assertEquals(" ", Highlighter.highlight(" ", "", false));
  }

  @Test
  public void testHighlightTree() throws Exception {
    final String s = Highlighter.highlight("путина putina путиной", tree, true);
    assertEquals("\u0007[путина\u0007] \u0007[putina\u0007] \u0007[путиной\u0007]", s);
  }

  @Test
  public void testExplicitDestroy() throws NoSuchFieldException, IllegalAccessException {
    final Highlighter highlighter = Highlighter.createByTree(tree);
    assertNotEquals(highlighter.getImplPointer(), 0);
    final String s = highlighter.highlight("путина putina путиной");
    assertEquals("\u0007[путина\u0007] \u0007[putina\u0007] \u0007[путиной\u0007]", s);
    highlighter.destroy();
    assertEquals(highlighter.getImplPointer(), 0);
  }

  @Test
  public void testHighlight() throws Exception {
    assertEquals("a", Highlighter.highlight("a", "b", false));
    assertEquals("\u0007[a\u0007]", Highlighter.highlight("a", "a", false));
    assertEquals("\u0007[a\u0007]", Highlighter.highlight("a", "A", false));
    assertEquals("\u0007[A\u0007]", Highlighter.highlight("A", "a", false));
    assertEquals("\u0007[a\u0007] b c", Highlighter.highlight("a b c", "a", false));
    assertEquals("\u0007[a\u0007]", Highlighter.highlight("a", "!a", false));
    assertEquals("\u0007[a\u0007] c c c \u0007[b\u0007]", Highlighter.highlight("a c c c b", "!a ~b", false));
    assertEquals("\u0007[a\u0007] c c c \u0007[b\u0007]", Highlighter.highlight("a c c c b", "!a && b", false));
  }

  @Test
  public void testCutNull() throws Exception {
    exception.expect(IllegalArgumentException.class);
    Highlighter.cut(null, 0);
  }

  @Test
  public void testCutEmpty() throws Exception {
    assertEquals("", Highlighter.cut("", 0));
    assertEquals("", Highlighter.cut("", 1));
    assertEquals("", Highlighter.cut("", 256));
  }

  @Test
  public void testCutZero() throws Exception {
    assertEquals("", Highlighter.cut("a", 0));
    assertEquals("", Highlighter.cut("aa", 0));
    assertEquals("", Highlighter.cut("aaa", 0));
    assertEquals("", Highlighter.cut("", 1));
  }

  @Test
  public void testCut() throws Exception {
    assertEquals("Данные...", Highlighter.cut("Данные для блока приходят в виде массива наборов значений.", 9));
    assertEquals("Данные для...", Highlighter.cut("Данные для блока приходят в виде массива наборов значений.", 13));
    assertEquals("Данные для блока приходят в виде массива наборов...", Highlighter.cut("Данные для блока приходят в виде массива наборов значений.", 57));
    assertEquals("Данные для блока приходят в виде массива наборов значений.", Highlighter.cut("Данные для блока приходят в виде массива наборов значений.", 59));
  }

  @Test
  public void testMakeTitleMargins() throws Exception {
    assertEquals("", Highlighter.makeTitle("Данные для блока приходят в виде массива наборов значений.", 0, 1f));
    assertEquals("", Highlighter.makeTitle("Данные для блока приходят в виде массива наборов значений.", Integer.MAX_VALUE, (float) Integer.MAX_VALUE));
    assertEquals("", Highlighter.makeTitle("Данные для блока приходят в виде массива наборов значений.", 0, (float) Integer.MAX_VALUE));
  }

  @Test
  public void testMakeTitle() throws Exception {
    final int[] lens = {240 - 24, 320 - 30, 420 - 40};
    assertEquals("А у нас есть...", Highlighter.makeTitle("А у нас есть планы по переходу на С++14?", 240 - 24, 22.5f));
    assertEquals("А у нас есть планы по...", Highlighter.makeTitle("А у нас есть планы по переходу на С++14?", 320 - 30, 22.5f));
    assertEquals("А у нас есть планы по переходу...", Highlighter.makeTitle("А у нас есть планы по переходу на С++14?", 420 - 40, 22.5f));
  }

  @Test
  public void textMakeQTree() {
    assertEquals("cHic40rh4uFgEGCQ4FRg0GAyYBBiycgvLpFiUGLQ4lTKSc0rSdQrKjVisOLhYAEqYgAqYjBgcGDwYAhgiGBIYMhgnMDIMIsRrGkRI0LHJkaGvYwMQHCCkeECY4oBgwWjkwTIHpDJGowGjAgDqhgaGBmApgAAl2kY7Q,,", Highlighter.makeQTree("host:lenta.ru"));
  }
}
