package ru.yandex.common.util.htmlstripper;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;

/**
 * @author imakunin
 */
public final class HtmlStripper {

  private final static Logger LOGGER = LoggerFactory.getLogger(HtmlStripper.class);

  static {
    try {
      loadLibrary("htmlstripper");
    } catch (IOException e) {
      throw new UnsatisfiedLinkError(e.getMessage());
    }
    registerNatives();
  }


  private HtmlStripper() {
  }

  private static void loadLibrary(final String name) throws IOException {
    LOGGER.info("Loading library htmlstripper");
    try {
      System.loadLibrary(name);
      return;
    } catch (UnsatisfiedLinkError ignored) {
    }

    try {
      final String libFileName = System.mapLibraryName(name);

      final File file;
      try (InputStream in = HtmlStripper.class.getClassLoader().getResourceAsStream(libFileName)) {
        if (in == null) {
          throw new IOException("Missing JNI library - " + libFileName);
        }

        final int pos = libFileName.lastIndexOf('.');
        file = File.createTempFile(libFileName.substring(0, pos), libFileName.substring(pos));
        file.deleteOnExit();
        Files.copy(in, file.toPath(), StandardCopyOption.REPLACE_EXISTING);
      }

      tmpFilename = file.getAbsolutePath();
      System.load(tmpFilename);
    } catch (Throwable e) {
      LOGGER.error("failed", e);
    }
  }

  private static native void registerNatives();

  public static native String strip(final String text);

  private native void destroy();

  private static volatile String tmpFilename = null;

  @Override
  protected void finalize() throws Throwable {
    destroy();
    if (tmpFilename != null) {
      Files.deleteIfExists(Paths.get(tmpFilename));
    }
    super.finalize();
  }
}
