package ru.yandex.kernel.inflector;

import java.nio.charset.StandardCharsets;

import javax.annotation.Nullable;

import com.sun.jna.Library;
import com.sun.jna.Native;


public class Inflector {
    private static final int MAX_RESULT_BYTES = 1000;

    private static LibApi libApi;
    static {
        libApi = Native.loadLibrary("inflector", LibApi.class);
    }

    private interface LibApi extends Library {
        int pluralize(String text, int targetNumber, int targetCase, int lang, boolean isAnimated, byte[] dst, int dstLen);
        int singularize(String text, int sourceNumber, int lang, byte[] dst, int dstLen);
    }

    @Nullable
    public static String pluralize(String text, int targetNumber) {
        return pluralizeImpl(text, targetNumber, Grammar.gNominative, Language.LANG_RUS, false);
    }

    @Nullable
    public static String pluralize(String text, int targetNumber, Language lang) {
        return pluralizeImpl(text, targetNumber, Grammar.gNominative, lang, false);
    }

    @Nullable
    public static String pluralize(String text, int targetNumber, Grammar targetCase) {
        return pluralizeImpl(text, targetNumber, targetCase, Language.LANG_RUS, false);
    }

    @Nullable
    public static String pluralize(String text, int targetNumber, Grammar targetCase, Language lang) {
        return pluralizeImpl(text, targetNumber, targetCase, lang, false);
    }

    @Nullable
    public static String pluralize(String text, int targetNumber, Grammar targetCase, Language lang, boolean isAnimated) {
        return pluralizeImpl(text, targetNumber, targetCase, lang, isAnimated);
    }

    @Nullable
    public static String singularize(String text, int sourceNumber) {
        return singularizeImpl(text, sourceNumber, Language.LANG_RUS);
    }

    @Nullable
    public static String singularize(String text, int sourceNumber, Language lang) {
        return singularizeImpl(text, sourceNumber, lang);
    }

    @Nullable
    private static String pluralizeImpl(String text, int targetNumber, Grammar targetCase, Language lang, boolean isAnimated) {
        byte[] buffer = new byte[MAX_RESULT_BYTES];
        int resultSize = libApi.pluralize(text, targetNumber, targetCase.getValue(), lang.getValue(), isAnimated, buffer, MAX_RESULT_BYTES);
        if (resultSize == 0) {
            return null;
        }
        return new String(buffer, 0, resultSize, StandardCharsets.UTF_8);
    }

    @Nullable
    private static String singularizeImpl(String text, int sourceNumber, Language lang) {
        byte[] buffer = new byte[MAX_RESULT_BYTES];
        int resultSize = libApi.singularize(text, sourceNumber, lang.getValue(), buffer, MAX_RESULT_BYTES);
        if (resultSize == 0) {
            return null;
        }
        return new String(buffer, 0, resultSize, StandardCharsets.UTF_8);
    }
}

