package ru.yandex.kernel.inflector;

import org.junit.jupiter.api.Test;

import java.util.ArrayList;
import java.util.Arrays;

import static org.junit.jupiter.api.Assertions.assertEquals;

//copy from https://a.yandex-team.ru/arc_vcs/kernel/inflectorlib/pluralize/pluralize_ut.cpp
public class InflectorTest {

    private void testPluralizeRussianWord(String word, String word2, String word5) {
        for (int n: Arrays.asList(1, 21, 1001)) {
            assertEquals(word, Inflector.pluralize(word, n, Grammar.gNominative, Language.LANG_RUS, true));
        }

        for (int n : Arrays.asList(2, 3, 4, 33, 44)) {
            assertEquals(word2, Inflector.pluralize(word, n, Grammar.gNominative, Language.LANG_RUS, true));
        }

        for (int n : Arrays.asList(0, 5, 6, 7, 8, 9, 10, 11, 12, 17, 20, 26, 38, 50, 100, 111, 256)) {
            assertEquals(word5, Inflector.pluralize(word, n, Grammar.gNominative, Language.LANG_RUS, true));
        }
    }

    private void testSingularizeRussianWord(String word, String word2, String word5) {
        for (int n : Arrays.asList(1, 21, 1001)) {
            assertEquals(word, Inflector.singularize(word, n, Language.LANG_RUS));
        }

        for (int n: Arrays.asList(2, 3, 4, 33, 44)) {
            assertEquals(word, Inflector.singularize(word2, n, Language.LANG_RUS));
        }

        for (int n: Arrays.asList(0, 5, 6, 7, 8, 9, 10, 11, 12, 17, 20, 26, 38, 50, 100, 111, 256)) {
            assertEquals(word, Inflector.singularize(word5, n, Language.LANG_RUS));
        }
    }

    static int[][] numbers = {
        {0},
        {5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 17, 19, 20, 77, 128},
        {1, 21, 31, 51, 71, 91, 101, 121, 131, 241, 391, 401, 431, 441, 491},
        {2, 3, 4, 22, 54, 73, 92, 93, 94, 102, 103, 104, 202, 444},
    };

    static int[][] numbersAnimal = {
        {0},
        {5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 17, 19, 20, 77, 128},
        {1, 21, 31, 51, 71, 91, 101, 121, 131, 241, 391, 401, 431, 441, 491},
        {2, 3, 4},
        {22, 54, 73, 92, 93, 94, 102, 103, 104, 202, 444},
    };

    private void testPluralizeRussianWordWithCaseInanimated(
        String[] wordSet,
        String wordSingle,
        int index
    ) {
        String wordNom = wordSet[0];
        String wordGen = wordSet[1];
        String wordDat = wordSet[2];
        String wordAcc = wordSet[3];
        String wordIns = wordSet[4];
        String wordAbl = wordSet[5];

         for (int n : numbers[index]) {
             assertEquals(wordNom, Inflector.pluralize(wordSingle, n, Grammar.gNominative, Language.LANG_RUS));
             assertEquals(wordGen, Inflector.pluralize(wordSingle, n, Grammar.gGenitive, Language.LANG_RUS));
             assertEquals(wordDat, Inflector.pluralize(wordSingle, n, Grammar.gDative, Language.LANG_RUS));
             assertEquals(wordAcc, Inflector.pluralize(wordSingle, n, Grammar.gAccusative, Language.LANG_RUS));
             assertEquals(wordIns, Inflector.pluralize(wordSingle, n, Grammar.gInstrumental, Language.LANG_RUS));
             assertEquals(wordAbl, Inflector.pluralize(wordSingle, n, Grammar.gAblative, Language.LANG_RUS));
         }
    }

    static void testPluralizeRussianWordWithCaseAnimated(
        String[] wordSet,
        String wordSingle,
        int index
    ) {
        String wordNom = wordSet[0];
        String wordGen = wordSet[1];
        String wordDat = wordSet[2];
        String wordAcc = wordSet[3];
        String wordIns = wordSet[4];
        String wordAbl = wordSet[5];

        for (int n : numbersAnimal[index]) {
            assertEquals(wordNom, Inflector.pluralize(wordSingle, n, Grammar.gNominative, Language.LANG_RUS, true));
            assertEquals(wordGen, Inflector.pluralize(wordSingle, n, Grammar.gGenitive, Language.LANG_RUS, true));
            assertEquals(wordDat, Inflector.pluralize(wordSingle, n, Grammar.gDative, Language.LANG_RUS, true));
            assertEquals(wordAcc, Inflector.pluralize(wordSingle, n, Grammar.gAccusative, Language.LANG_RUS, true));
            assertEquals(wordIns, Inflector.pluralize(wordSingle, n, Grammar.gInstrumental, Language.LANG_RUS, true));
            assertEquals(wordAbl, Inflector.pluralize(wordSingle, n, Grammar.gAblative, Language.LANG_RUS, true));
        }
    }

    static String[][] words = {
        {"попугай", "попугая", "попугаев"},
        {"ночь", "ночи", "ночей"},
        {"корова", "коровы", "коров"},
        {"рубль", "рубля", "рублей"},
        {"ребенок", "ребенка", "детей"},
        {"веселый чиж", "веселых чижа", "веселых чижей"},
        {"мгновение весны", "мгновения весны", "мгновений весны"},
        {"бакинский комиссар", "бакинских комиссара", "бакинских комиссаров"},
        {"московский комсомолец", "московских комсомольца", "московских комсомольцев"},
        {"лежачий полицейский", "лежачих полицейских", "лежачих полицейских"},
        {"танкист - веселый друг", "танкиста - веселых друга", "танкистов - веселых друзей"},
        //{"стакан муки", "стакана муки", "стаканов муки"},
        //{"чайная ложка сахара", "чайные ложки сахара", "чайных ложек сахара"},
        //{"оттенок серого", "оттенка серого", "оттенков серого"},
    };

    static String[][][] wordsInanimated = {
        {
            {"дней", "дней", "дней", "дней", "дней", "дней"},
            {"дней", "дней", "дням", "дней", "днями", "днях"},
            {"день", "дня", "дню", "день", "днем", "дне"},
            {"дня", "дней", "дням", "дня", "днями", "днях"},
        },
        {
            {"желтых автобусов", "желтых автобусов", "желтых автобусов", "желтых автобусов", "желтых автобусов", "желтых автобусов"}, //= 0
            {"желтых автобусов", "желтых автобусов", "желтым автобусам", "желтых автобусов", "желтыми автобусами", "желтых автобусах"}, // mod10 : 0, 5..9; mod100 : 11..14
            {"желтый автобус", "желтого автобуса", "желтому автобусу", "желтый автобус", "желтым автобусом", "желтом автобусе"}, // mod10 : 1
            {"желтых автобуса", "желтых автобусов", "желтым автобусам", "желтых автобуса", "желтыми автобусами", "желтых автобусах"}, // mod10: 2..4
        },
        {
            {"белых стульев", "белых стульев", "белых стульев", "белых стульев", "белых стульев", "белых стульев"},
            {"белых стульев", "белых стульев", "белым стульям", "белых стульев", "белыми стульями", "белых стульях"},
            {"белый стул", "белого стула", "белому стулу", "белый стул", "белым стулом", "белом стуле"},
            {"белых стула", "белых стульев", "белым стульям", "белых стула", "белыми стульями", "белых стульях"},
        },
        {
            {"ярких ламп", "ярких ламп", "ярких ламп", "ярких ламп", "ярких ламп", "ярких ламп"},
            {"ярких ламп", "ярких ламп", "ярким лампам", "ярких ламп", "яркими лампами", "ярких лампах"},
            {"яркая лампа", "яркой лампы", "яркой лампе", "яркую лампу", "яркой лампой", "яркой лампе"},
            {"яркие лампы", "ярких ламп", "ярким лампам", "яркие лампы", "яркими лампами", "ярких лампах"},
        },
        {
            {"больших вселенных", "больших вселенных", "больших вселенных", "больших вселенных", "больших вселенных", "больших вселенных"},
            {"больших вселенных", "больших вселенных", "большим вселенным", "больших вселенных", "большими вселенными", "больших вселенных"},
            {"большая вселенная", "большой вселенной", "большой вселенной", "большую вселенную", "большой вселенной", "большой вселенной"},
            {"большие вселенные", "больших вселенных", "большим вселенным", "большие вселенные", "большими вселенными", "больших вселенных"},
        },
        {
            {"лежачих полицейских", "лежачих полицейских", "лежачих полицейских", "лежачих полицейских", "лежачих полицейских", "лежачих полицейских"},
            {"лежачих полицейских", "лежачих полицейских", "лежачим полицейским", "лежачих полицейских", "лежачими полицейскими", "лежачих полицейских"},
            {"лежачий полицейский", "лежачего полицейского", "лежачему полицейскому", "лежачий полицейский", "лежачим полицейским", "лежачем полицейском"},
            {"лежачих полицейских", "лежачих полицейских", "лежачим полицейским", "лежачих полицейских", "лежачими полицейскими", "лежачих полицейских"},
        },
        // {
        //     {"городов-героев", "городов-героев", "городов-героев", "городов-героев", "городов-героев", "городов-героев"},
        //     {"городов-героев", "городов-героев", "городам-героям", "городов-героев", "городами-героями", "городах-героях"},
        //     {"город-герой", "города-героя", "городу-герою", "город-герой", "городом-героем", "городе-герое"},
        //     {"города-героя", "городов-героев", "городам-героям", "города-героя", "городами-героями", "городах-героях"},
        // },
    };

    static String[] wordsInanimatedSingNom = {
        "день",
        "желтый автобус",
        "белый стул",
        "яркая лампа",
        "большая вселенная",
        "лежачий полицейский",
        // "город-герой",
    };

    static String[][][] wordsAnimated = {
        {
            {"котов", "котов", "котов", "котов", "котов", "котов"},
            {"котов", "котов", "котам", "котов", "котами", "котах"},
            {"кот", "кота", "коту", "кота", "котом", "коте"},
            {"кота", "котов", "котам", "котов", "котами", "котах"},
            {"кота", "котов", "котам", "кота", "котами", "котах"},
        },
        {
            {"веселых детей", "веселых детей", "веселых детей", "веселых детей", "веселых детей", "веселых детей"}, //= 0
            {"веселых детей", "веселых детей", "веселым детям", "веселых детей", "веселыми детьми", "веселых детях"}, // mod10 : 0, 5..9; mod100 : 11..14
            {"веселый ребенок", "веселого ребенка", "веселому ребенку", "веселого ребенка", "веселым ребенком", "веселом ребенке"}, //1,21,31,41,...
            {"веселых ребенка", "веселых детей", "веселым детям", "веселых детей", "веселыми детьми", "веселых детях"}, //  2..4
            {"веселых ребенка", "веселых детей", "веселым детям", "веселых ребенка", "веселыми детьми", "веселых детях"}, //mod100 : 22..24, 32..34, ...

        },
        {
            {"добрых псов", "добрых псов", "добрых псов", "добрых псов", "добрых псов", "добрых псов"},
            {"добрых псов", "добрых псов", "добрым псам", "добрых псов", "добрыми псами", "добрых псах"},
            {"добрый пёс", "доброго пса", "доброму псу", "доброго пса", "добрым псом", "добром псе"},
            {"добрых пса", "добрых псов", "добрым псам", "добрых псов", "добрыми псами", "добрых псах"},
            {"добрых пса", "добрых псов", "добрым псам", "добрых пса", "добрыми псами", "добрых псах"},
        },
        {
            {"певчих птиц", "певчих птиц", "певчих птиц", "певчих птиц", "певчих птиц", "певчих птиц"},
            {"певчих птиц", "певчих птиц", "певчим птицам", "певчих птиц", "певчими птицами", "певчих птицах"},
            {"певчая птица", "певчей птицы", "певчей птице", "певчую птицу", "певчей птицей", "певчей птице"},
            {"певчие птицы", "певчих птиц", "певчим птицам", "певчих птиц", "певчими птицами", "певчих птицах"},
            {"певчие птицы", "певчих птиц", "певчим птицам", "певчие птицы", "певчими птицами", "певчих птицах"},
        },
        {
            {"красивых горничных", "красивых горничных", "красивых горничных", "красивых горничных", "красивых горничных", "красивых горничных"},
            {"красивых горничных", "красивых горничных", "красивым горничным", "красивых горничных", "красивыми горничными", "красивых горничных"},
            {"красивая горничная", "красивой горничной", "красивой горничной", "красивую горничную", "красивой горничной", "красивой горничной"},
            {"красивые горничные", "красивых горничных", "красивым горничным", "красивых горничных", "красивыми горничными", "красивых горничных"},
            {"красивые горничные", "красивых горничных", "красивым горничным", "красивые горничные", "красивыми горничными", "красивых горничных"},
        },
        {
            {"веб-разработчиков", "веб-разработчиков", "веб-разработчиков", "веб-разработчиков", "веб-разработчиков", "веб-разработчиков"},
            {"веб-разработчиков", "веб-разработчиков", "веб-разработчикам", "веб-разработчиков", "веб-разработчиками", "веб-разработчиках"},
            {"веб-разработчик", "веб-разработчика", "веб-разработчику", "веб-разработчика", "веб-разработчиком", "веб-разработчике"},
            {"веб-разработчика", "веб-разработчиков", "веб-разработчикам", "веб-разработчиков", "веб-разработчиками", "веб-разработчиках"},
            {"веб-разработчика", "веб-разработчиков", "веб-разработчикам", "веб-разработчика", "веб-разработчиками", "веб-разработчиках"},
        },
        {
            {"слесарей-сантехников", "слесарей-сантехников", "слесарей-сантехников", "слесарей-сантехников", "слесарей-сантехников", "слесарей-сантехников"},
            {"слесарей-сантехников", "слесарей-сантехников", "слесарям-сантехникам", "слесарей-сантехников", "слесарями-сантехниками", "слесарях-сантехниках"},
            {"слесарь-сантехник", "слесаря-сантехника", "слесарю-сантехнику", "слесаря-сантехника", "слесарем-сантехником", "слесаре-сантехнике"},
            {"слесаря-сантехника", "слесарей-сантехников", "слесарям-сантехникам", "слесарей-сантехников", "слесарями-сантехниками", "слесарях-сантехниках"},
            {"слесаря-сантехника", "слесарей-сантехников", "слесарям-сантехникам", "слесаря-сантехника", "слесарями-сантехниками", "слесарях-сантехниках"},
        }
    };

    static String[] wordsAnimatedSingNom = {
        "кот",
        "веселый ребенок",
        "добрый пёс",
        "певчая птица",
        "красивая горничная",
        "веб-разработчик",
        "слесарь-сантехник",
    };

    @Test
    public void pluralize_russian_words_test() {
        for (String[] wordSet : words) {
            testPluralizeRussianWord(wordSet[0], wordSet[1], wordSet[2]);
        }
    }

    @Test
    public void singularize_russian_words_test() {
        for (String[] wordSet : words) {
            testSingularizeRussianWord(wordSet[0], wordSet[1], wordSet[2]);
        }
    }

    @Test
    public void pluralize_russian_words_with_case_inanimated() {
        int j = 0;
        for (String[][] wordSet : wordsInanimated) {
            int i = 0;
            for (String[] wordCase : wordSet) {
                testPluralizeRussianWordWithCaseInanimated(wordCase, wordsInanimatedSingNom[j], i);
                i++;
            }
            j++;
        }
    }

    @Test
    public void pluralize_russian_words_with_case_animated() {
        int j = 0;
        for (String[][] wordSet : wordsAnimated) {
            int i = 0;
            for (String[] wordCase : wordSet) {
                testPluralizeRussianWordWithCaseAnimated(wordCase, wordsAnimatedSingNom[j], i);
                i++;
            }
            j++;
        }
    }
}
