#include "ru_yandex_common_util_lemmer_Lemmer.h"

#include <kernel/lemmer/core/language.h>

#include <util/generic/set.h>
#include <util/string/strip.h>

namespace LemmerJniBindings {

    //stores data required for jni interaction
    struct TJniData {

        jclass StringClass;
        jclass LemmaClass;
        jclass FormClass;
        jclass QualityClass;
        jclass GrammarClass;

        jmethodID FormConstructor;
        jmethodID LemmaConstructor;
        jmethodID GetQualityByCode;
        jmethodID GetGrammarByCode;

        JNIEnv *Env;

        //FIXME: the constants should be extracted
        TJniData(JNIEnv *env) : Env(env) {

            StringClass = env->FindClass("java/lang/String");
            LemmaClass = env->FindClass("ru/yandex/common/util/lemmer/Lemma");
            FormClass = env->FindClass("ru/yandex/common/util/lemmer/Lemma$Form");
            QualityClass = env->FindClass("ru/yandex/common/util/lemmer/Quality");
            GrammarClass = env->FindClass("ru/yandex/common/util/lemmer/Grammar");

            FormConstructor = env->GetMethodID(FormClass, "<init>", "(Ljava/lang/String;[Lru/yandex/common/util/lemmer/Grammar;)V");
            LemmaConstructor = env->GetMethodID(LemmaClass, "<init>", "(Lru/yandex/common/util/lemmer/Quality;Ljava/lang/String;[Lru/yandex/common/util/lemmer/Lemma$Form;[Lru/yandex/common/util/lemmer/Grammar;[Lru/yandex/common/util/lemmer/Lemma$Form;)V");
            GetQualityByCode = env->GetStaticMethodID(QualityClass, "getByCode", "(I)Lru/yandex/common/util/lemmer/Quality;");
            GetGrammarByCode = env->GetStaticMethodID(GrammarClass, "getGrammarById", "(C)Lru/yandex/common/util/lemmer/Grammar;");

        }
    };

    //Takes an array of grammar codes on the input and returns a java Grammar[]
    static jobjectArray GetJavaGrammars( const TJniData& jni,  char const * gramms, int grammDim) {
        jobjectArray result = (jobjectArray) jni.Env->NewObjectArray(grammDim, jni.GrammarClass, nullptr);

        for(int i = 0; i < grammDim; i ++) {
            jobject gramm = jni.Env->CallStaticObjectMethod(jni.GrammarClass, jni.GetGrammarByCode, (unsigned char)gramms[i]);
            jni.Env->SetObjectArrayElement(result, i, gramm);
        }

        return result;
    }

    //filters the forms with accordance to the given allowed_grammars
    static void GetFilteredForms(const TJniData& jni, const TSet<char>& allowedGrammars, const char* const* gramms, int grammNumber, const TUtf16String& text, TVector<jobject>& forms) {
        jstring word = jni.Env->NewStringUTF(WideToUTF8(text).data());

        for(int pos = 0; pos < grammNumber; pos ++) {
            bool formWillBeUsed = false;
            int grammDim = 0;

            for(unsigned i = 0; gramms[pos][i]; i++) {
                unsigned char currGramm = gramms[pos][i];
                if(allowedGrammars.count(currGramm) > 0 || allowedGrammars.empty()) {
                    formWillBeUsed = true;
                }
                grammDim ++;
            }

            if(formWillBeUsed) {

                jobjectArray javaGrammars = GetJavaGrammars(jni, gramms[pos], grammDim);

                jobject form = jni.Env->NewObject(jni.FormClass, jni.FormConstructor, word, javaGrammars);

                forms.push_back(form);

            }

        }

    }

    //converts a TVector of Forms's to java Form[]
    static jobjectArray ConvertFormsToSingleArray(const TJniData& jni, const TVector<jobject>& forms) {

        jobjectArray result = (jobjectArray) jni.Env->NewObjectArray(forms.size(), jni.FormClass, nullptr);

        for (unsigned i = 0; i < forms.size(); ++i) {
            jni.Env->SetObjectArrayElement(result, i, forms[i]);
        }

        return result;
    }

    //filters lemmas according to qualities and to their uniqueness
    static void FilterLemmas(const TWLemmaArray& lemmas, TWUniqueLemmaArray& uniques, int qualities) {
        const TYandexLemma * data = lemmas.data();
        size_t len = lemmas.size();


        uniques.resize(0);
        uniques.reserve(len);

        const TYandexLemma* dataend = data + len;
        for (const TYandexLemma* p1 = data; p1 != dataend; p1++)  {
            if(p1->GetQuality() == 0 || (qualities & p1->GetQuality())) {
                uniques.push_back(p1);
            }
        }

    }

    //actually analyzes the word given, returns Lemma[]
    static jobjectArray AnalyzeWord(const TUtf16String& query, ELanguage language, int qualities, TSet<char> allowedGramms, const TJniData& jni) {

        TWLemmaArray lemmas;
        NLemmer::TAnalyzeWordOpt opt("", NLemmer::SfxOnly, NLemmer::MtnSplitAllPossible, NLemmer::AccFoundling);
        NLemmer::AnalyzeWord(query.data(), query.size(), lemmas, TLangMask( language ), nullptr, opt);

        TWUniqueLemmaArray filtered;
        FilterLemmas(lemmas, filtered, qualities);

        TVector<jobject> resultVector;
        for(unsigned currLemma = 0; currLemma < filtered.size(); currLemma ++) {

            const TYandexLemma* lemma = filtered[currLemma];

            jstring initialForm = jni.Env->NewStringUTF(WideToUTF8(lemma->GetText(), lemma->GetTextLength()).data());

            //here we get the possible initial forms
            TVector<jobject> flexFormsVector;
            GetFilteredForms(jni, TSet<char>(), lemma->GetFlexGram(), lemma->FlexGramNum(), query, flexFormsVector);

            jobjectArray flexForms = ConvertFormsToSingleArray(jni, flexFormsVector);

            //here we get the lemma's quality
            jobject quality = jni.Env->CallStaticObjectMethod(jni.QualityClass, jni.GetQualityByCode, lemma->GetQuality());


            //here we get the stem gram
            const char * stemGram = lemma->GetStemGram();

            int stemDim = 0;
            for(int i = 0; stemGram[i]; i++) {
                stemDim++;
            }

            jobjectArray stem = GetJavaGrammars(jni, stemGram, stemDim);

            //here we get all forms
            TWordformArray forms;
            NLemmer::Generate(*lemma, forms);

            TVector<jobject> allForms;

            for (TWordformArray::const_iterator i = forms.begin(), mi = forms.end(); i != mi; ++i) {
                TVector<jobject> currForms;
                GetFilteredForms(jni, allowedGramms, i->GetFlexGram(), i->FlexGramNum(), i->GetText(), currForms);

                allForms.insert(allForms.end(), currForms.begin(), currForms.end());
            }

            jobject lemmaResult = jni.Env->NewObject(jni.LemmaClass, jni.LemmaConstructor, quality, initialForm, flexForms, stem, ConvertFormsToSingleArray(jni, allForms));

            resultVector.push_back(lemmaResult);
        }

        jobjectArray result = (jobjectArray) jni.Env->NewObjectArray(resultVector.size(), jni.LemmaClass, nullptr);

        for(unsigned i = 0; i < resultVector.size(); i ++) {
            jni.Env->SetObjectArrayElement(result, i, resultVector[i]);
        }


        return result;
    }

}


JNIEXPORT jobjectArray JNICALL Java_ru_yandex_common_util_lemmer_Lemmer_jniAnalyzeWord
  (JNIEnv *env, jclass, jobjectArray queries, jstring lang, jint qualities, jcharArray grammars) {

    LemmerJniBindings::TJniData jni(env);

    const char *languageCstring = env->GetStringUTFChars(lang, nullptr);
    ELanguage language = LanguageByName(languageCstring);

    jsize grammsLen = env->GetArrayLength(grammars);

    TSet<char> allowedGramms;
    jchar *grammarJids = env->GetCharArrayElements(grammars, nullptr);
    for(int i = 0; i < grammsLen; i ++) {
        allowedGramms.insert(grammarJids[i]);
    }


    jsize queriesCount = env->GetArrayLength(queries);

    jobjectArray sampleArray = (jobjectArray) env->NewObjectArray(0, jni.LemmaClass, nullptr);
    jobjectArray result = (jobjectArray) env->NewObjectArray(queriesCount, env->GetObjectClass(sampleArray), nullptr);
    for(int i = 0; i < queriesCount; i ++) {
        jstring query = (jstring) env->GetObjectArrayElement(queries, i);
        const char *queryCstring = env->GetStringUTFChars(query, nullptr);
        TUtf16String queryWtroka = UTF8ToWide(queryCstring);
        StripString(queryWtroka);

        jobjectArray lemmaResult = LemmerJniBindings::AnalyzeWord(queryWtroka, language, qualities, allowedGramms, jni);

        env->SetObjectArrayElement(result, i, lemmaResult);

        env->ReleaseStringUTFChars(query, queryCstring);
    }

    env->ReleaseCharArrayElements(grammars, grammarJids, 0);

    env->ReleaseStringUTFChars(lang, languageCstring);

    return result;
}
