package ru.yandex.kernel.hosts.owner;

import java.nio.charset.StandardCharsets;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sun.jna.Library;
import com.sun.jna.Native;
import com.sun.jna.Pointer;

public class OwnerCanonizerImpl implements OwnerCanonizer {
    private static final int MAX_RESULT_BYTES = 2_000;

    private static LibApi libApi;
    static {
        libApi = Native.loadLibrary("owner_canonizer", LibApi.class);
    }

    private final Pointer canonizer;

    public OwnerCanonizerImpl() {
        Pointer ptr = libApi.InitOwnerCanonizer();
        if (Objects.equals(Pointer.NULL, ptr)) {
            throw new RuntimeException("Cannot initialize OwnerCanonizer");
        }
        this.canonizer = ptr;
    }

    private interface LibApi extends Library {
        Pointer InitOwnerCanonizer();
        void DeleteOwnerCanonizer(Pointer canonizer);
        void DeleteOwnerPartExtractor(Pointer extractor);
        long GetUrlOwner(Pointer canonizer, String url, byte[] dst, long dstLen);
    }

    @Override
    @Nullable
    public String getUrlOwner(@Nonnull String url) {
        byte[] buffer = new byte[MAX_RESULT_BYTES];
        long resultSize = libApi.GetUrlOwner(this.canonizer, url.toLowerCase(), buffer, MAX_RESULT_BYTES);
        if (resultSize == 0) {
            return null;
        }
        return new String(buffer, 0, (int)resultSize, StandardCharsets.UTF_8);
    }

    @Override
    protected void finalize() throws Throwable {
        assert !Objects.equals(Pointer.NULL, this.canonizer);
        libApi.DeleteOwnerCanonizer(this.canonizer);
        super.finalize();
    }

}
