#include <util/generic/string.h>
#include <util/generic/buffer.h>
#include <library/cpp/resource/resource.h>
#include <kernel/signurl/signurl.h>

#include "ru_yandex_common_signurl_Signurl.h"

jclass getClassReference(JNIEnv *env, const char *classname) {
    jclass tempLocalClassRef = env->FindClass(classname);
    jclass globalRef = (jclass) env->NewGlobalRef(tempLocalClassRef);
    env->DeleteLocalRef(tempLocalClassRef);
    return globalRef;
}

static jint JNI_VERSION = JNI_VERSION_1_8;
static jclass ireClass;
static jclass stringClass;
static jstring emptyString;

class TWrappedSignurl {
public:
    explicit TWrappedSignurl(const TString& base64EncodedKey, ui8 keyNo = 1)
        : KeyNo_(keyNo)
    {
        NSignUrl::InitKey(Key_, NSignUrl::Base64PadDecode(base64EncodedKey));
    }
    TString CryptPrependKeyNo(const TString& data) {
        TString out = Crypt(data);
        out.prepend(KeyNo_);
        return Base64EncodeUrl(out);
    }
    TString DecryptWithKeyNo(const TString& data) {
        TString decoded(NSignUrl::Base64PadDecode(data));
        TBuffer buffer(NSignUrl::CryptedSize(decoded.Size() - 1));
        size_t len = NSignUrl::Decrypt(decoded.Data() + 1, decoded.Size() - 1, buffer.Data(), Key_);
        buffer.Resize(len);
        return TString(buffer.Data(), buffer.Size());
    }

private:
    TString Crypt(const TString& data) {
        size_t len = NSignUrl::CryptedSize(data.Size());
        TBuffer buffer(len);
        size_t crypted_len = NSignUrl::Crypt(data.Data(), buffer.Data(), Key_.second);
        buffer.Resize(crypted_len);
        return TString(buffer.Data(), buffer.Size());
    }

    NSignUrl::TKey Key_;
    ui8 KeyNo_;
};

#define GET_STR_ARG(arg)                                                  \
    TString arg;                                                          \
    do {                                                                  \
        if (Y_LIKELY(j##arg)) {                                           \
            const char* c##arg = env->GetStringUTFChars(j##arg, nullptr); \
            arg = TString(c##arg);                                        \
            env->ReleaseStringUTFChars(j##arg, c##arg);                   \
        }                                                                 \
    } while (0);

jint JNI_OnLoad(JavaVM* jvm, void*) {
    JNIEnv* env;
    if (jvm->GetEnv(reinterpret_cast<void**>(&env), JNI_VERSION) != JNI_OK) {
        return JNI_ERR;
    }
    ireClass = getClassReference(env, "java/lang/RuntimeException");
    stringClass = getClassReference(env, "java/lang/String");
    emptyString = (jstring) env->NewGlobalRef(env->NewStringUTF(nullptr));
    return JNI_VERSION;
}

void JNI_OnUnload(JavaVM *jvm, void *) {
    JNIEnv* env;
    jvm->GetEnv(reinterpret_cast<void**>(&env), JNI_VERSION);
    env->DeleteGlobalRef(ireClass);
    env->DeleteGlobalRef(stringClass);
    env->DeleteGlobalRef(emptyString);
}

jlong Java_ru_yandex_common_signurl_Signurl_create(JNIEnv* env, jclass, jstring jkey) {
    if (!jkey) {
        env->ThrowNew(ireClass, "Illegal key");
    }
    GET_STR_ARG(key);
    auto obj = new TWrappedSignurl(key);
    return (long)obj;
}

void Java_ru_yandex_common_uatraits_Uatraits_delete(JNIEnv*, jclass, jlong jptr) {
    auto obj = (TWrappedSignurl*)jptr;
    delete obj;
}

jstring JNICALL Java_ru_yandex_common_signurl_Signurl_crypt(JNIEnv* env, jobject, jlong jptr, jstring jdata) {
    auto obj = (TWrappedSignurl*)jptr;
    GET_STR_ARG(data);
    TString result = obj->CryptPrependKeyNo(data);
    return env->NewStringUTF(result.data());
}

jstring JNICALL Java_ru_yandex_common_signurl_Signurl_decrypt(JNIEnv* env, jobject, jlong jptr, jstring jdata) {
    auto obj = (TWrappedSignurl*)jptr;
    GET_STR_ARG(data);
    TString result = obj->DecryptWithKeyNo(data);
    return env->NewStringUTF(result.data());
}

#undef GET_STR_ARG
