package ru.yandex.common.uatraits;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Objects;
import java.util.Scanner;

import static org.hamcrest.CoreMatchers.containsString;
import static org.junit.Assert.*;

/**
 * @author zador
 * @date 2017-12-07 00:18
 */
public class UatraitsTest {
    private static final String EMPTY_PROFILES_CONFIG = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><profiles></profiles>";
    private static final String EXTRAS_CONFIG = getResourceAsString("test-extras.xml");

    @Rule
    public ExpectedException thrown = ExpectedException.none();
    private Uatraits uatraits;

    @Before
    public void setUp() throws Exception {
        uatraits = new Uatraits();
    }

    @Test
    public void testNull() throws Exception {
        final Map<String, String> result = uatraits.classify(null);
        assertEquals(6, result.size());
        assertTrue(result.containsKey("BrowserName"));
        assertEquals("Unknown", result.get("BrowserName"));
    }

    @Test
    public void testEmpty() throws Exception {
        final Map<String, String> result = uatraits.classify("");
        assertEquals(6, result.size());
        assertTrue(result.containsKey("BrowserName"));
        assertEquals("Unknown", result.get("BrowserName"));
    }

    @Test
    public void test() throws Exception {
        final Map<String, String> result = uatraits.classify("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_6) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/61.0.3163.100 YaBrowser/17.10.0.2052 Yowser/2.5 Safari/537.36");
        assertEquals(13, result.size());
        assertTrue(result.containsKey("BrowserName"));
        assertEquals("YandexBrowser", result.get("BrowserName"));
    }

    @Test
    public void testWithExtrasWrongConfigErrorTest() {
        try {
            uatraits = new Uatraits(EMPTY_PROFILES_CONFIG, ">/");
        } catch (InitializationException e) {
            assertThat(e.getMessage(), containsString("Start tag expected, '<' not found"));
        }
    }

    @Test
    public void testWithExtrasFirstFeatureTrue() throws InitializationException {
        uatraits = new Uatraits(EMPTY_PROFILES_CONFIG, EXTRAS_CONFIG);
        Map<String, String> classify = uatraits.classify("3gpp-gba Mozilla/4.0 (compatible; MSIE 5.0; S60/3.0 C503/1.0 Profile/MIDP-2.0 Configuration/CLDC-1.1)");
        assertTrue(classify.containsKey("firstTestFeature"));
        assertEquals("someValue", classify.get("firstTestFeature"));
    }

    @Test
    public void testWithExtrasFirstFeatureFalse() throws InitializationException {
        uatraits = new Uatraits(EMPTY_PROFILES_CONFIG, EXTRAS_CONFIG);
        Map<String, String> classify = uatraits.classify("Custom User-Agent=Mozilla/5.0 Gecko/20100101 Opera/9.80 Version/12.00 (Windows NT 6.2; Win64; MSIE 10.0)");
        assertFalse(classify.containsKey("firstTestFeature"));
    }

    private static String getResourceAsString(String fileName) {
        try(InputStream inputStream = UatraitsTest.class.getClassLoader().getResourceAsStream(fileName)) {
            Scanner scanner = new Scanner(Objects.requireNonNull(inputStream)).useDelimiter("\\A");
            return scanner.hasNext() ? scanner.next() : "";
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}