package ru.yandex.common.util.urlid;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;

/**
 * @author imakunin
 */
public final class UrlId {

  static {
    try {
      loadLibrary("urlid");
    } catch (IOException e) {
      throw new UnsatisfiedLinkError(e.getMessage());
    }
    registerNatives();
  }


  private UrlId() {
  }

  private static void loadLibrary(final String name) throws IOException {
    try {
      System.loadLibrary(name);
      return;
    } catch (UnsatisfiedLinkError ignored) {
    }

    final String libFileName = System.mapLibraryName(name);

    final File file;
    try (InputStream in = UrlId.class.getClassLoader().getResourceAsStream(libFileName)) {
      if (in == null) {
        throw new IOException("Missing JNI library - " + libFileName);
      }

      final int pos = libFileName.lastIndexOf('.');
      file = File.createTempFile(libFileName.substring(0, pos), libFileName.substring(pos));
      file.deleteOnExit();
      Files.copy(in, file.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    tmpFilename = file.getAbsolutePath();
    System.load(tmpFilename);
  }


  private static native void registerNatives();

  public static native String urlIdToDocId(final String url, final String lang) throws IllegalArgumentException;

  private native void destroy();

  private static volatile String tmpFilename = null;

  @Override
  protected void finalize() throws Throwable {
    destroy();
    if (tmpFilename != null) {
      Files.deleteIfExists(Paths.get(tmpFilename));
    }
    super.finalize();
  }
}
