#include <library/cpp/json/json_value.h>
#include <library/cpp/json/writer/json.h>
#include <library/cpp/json/yson/json2yson.h>
#include <library/cpp/yson/parser.h>
#include <library/cpp/yson/json/yson2json_adapter.h>

#include <util/datetime/cputimer.h>
#include <util/generic/stack.h>
#include <util/stream/file.h>
#include <util/string/builder.h>

#include <apphost/lib/converter/converter.h>

#ifdef __cplusplus
extern "C" {
#endif
#define NO_XSLOCKS
#include <EXTERN.h>
#include <perl.h>
#include <XSUB.h>
#ifdef __cplusplus
}
#endif

enum class BlessingFlags {
    ConvertBlessed = 0,
    AllowBlessed = 1,
    AllowUnknown = 2
};

inline bool isFlag(const i32 mask, const BlessingFlags flag) {
    return (mask >> static_cast<i32>(flag)) & 1;
}

inline void setFlag(i32& mask, const BlessingFlags flag) {
    mask |= (1 << static_cast<i32>(flag));
}

void WritePerlObject(SV* sv, NYT::TYson2JsonCallbacksAdapter* adapter, const i32 blessingMask, bool isOuter = false);

class IPerlObjectBuilder : public NYson::TYsonConsumerBase {
public:
    virtual SV* GetRoot() const = 0;
};

template<class TMakeSV, class TMakeHashKey>
class TPerlObjectBuilder : public IPerlObjectBuilder {

public:
    SV* GetRoot() const override {
        return ObjectsStack.empty() ? nullptr : ObjectsStack.top();
    }

    void OnStringScalar(TStringBuf value) override {
        SV* stringSv = TMakeSV::Make(value);
        ObjectsStack.push(stringSv);
        Update();
    }

    void OnInt64Scalar(i64 value) override {
        ObjectsStack.push(newSViv(value));
        Update();
    }

    void OnUint64Scalar(ui64 value) override {
        ObjectsStack.push(newSVuv(value));
        Update();
    }

    void OnDoubleScalar(double value) override {
        ObjectsStack.push(newSVnv(value));
        Update();
    }

    void OnBooleanScalar(bool value) override {
        HV* stash = gv_stashpv("JSON::PP::Boolean", true);
        ObjectsStack.push(sv_bless(newRV_inc(newSViv(value ? 1 : 0)), stash));
        Update();
    }

    void OnEntity() override {
        ObjectsStack.push(newSV_type(SVt_NULL));
        Update();
    }

    void OnBeginList() override {
        ObjectsStack.push((SV*) newAV());
    }

    void OnListItem() override {
    }

    void OnEndList() override {
        Update();
    }

    void OnBeginMap() override {
        ObjectsStack.push((SV*) newHV());
    }

    void OnKeyedItem(TStringBuf key) override {
        SV* svKey = TMakeHashKey::Make(key);
        ObjectsStack.push(svKey);
    }

    void OnEndMap() override {
        Update();
    }

    void OnBeginAttributes() override {
    }

    void OnEndAttributes() override {
    }

private:
    TStack<SV*> ObjectsStack;

    void Update() {
        if (ObjectsStack.empty()) {
            croak("Error! Big Problems while build Perl object!");
        } else if (ObjectsStack.size() == 1) {
            return;
        }

        SV* topObject = ObjectsStack.top();
        if (SvTYPE(topObject) == SVt_PVAV || SvTYPE(topObject) == SVt_PVHV) {
            topObject = newRV_noinc(topObject);
        }
        ObjectsStack.pop();

        if (SvTYPE(ObjectsStack.top()) == SVt_PVAV) {
            av_push((AV*) ObjectsStack.top(), topObject);
        } else if (SvTYPE(ObjectsStack.top()) == SVt_PV) {
            SV* key = ObjectsStack.top();
            ObjectsStack.pop();
            if (ObjectsStack.empty() || SvTYPE(ObjectsStack.top()) != SVt_PVHV) {
                croak("Error! Big Problems while build Perl object!");
            }
            hv_store_ent((HV*) ObjectsStack.top(), key, topObject, 0);
            SvREFCNT_dec(key);
        }
    }
};

class TMakeSV {
public:
    static SV* Make(const TStringBuf& value) {
        return newSVpvn(value.data(), value.length());
    }
};

class TMakeSVUtf8 {
public:
    static SV* Make(const TStringBuf& value) {
        return newSVpvn_flags(value.data(), value.length(), SVf_UTF8);
    }
};

static int ref_bool_type (SV *sv) {
    svtype svt = SvTYPE (sv);

    if (svt < SVt_PVAV) {
        STRLEN len = 0;
        char *pv = svt ? SvPV (sv, len) : 0;

        if (len == 1)
            if (*pv == '1')
                return 1;
            else if (*pv == '0')
                return 0;
    }

    return -1;
}

void HandleNamedHash(SV* sv, NYT::TYson2JsonCallbacksAdapter* adapter, const i32 blessingMask) {
    bool serialized = false;
    HV* hv = (HV*) sv;
    static const TString toJsonName = "TO_JSON";
    SV* keySv = newSVpv(toJsonName.c_str(), toJsonName.length());

    if (isFlag(blessingMask, BlessingFlags::ConvertBlessed)) {
        dSP;
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        EXTEND(SP, 1);
        PUSHs(sv_2mortal(newRV_inc(sv)));
        PUTBACK;

        int cnt = perl_call_pv((TStringBuilder() << sv_reftype(sv, true) << "::" << toJsonName).c_str(), G_SCALAR | G_EVAL);

        SPAGAIN;

        if (cnt == 1) {
            SV* resultObject = POPs;
            if (SvOK(resultObject)) {
                WritePerlObject(resultObject, adapter, blessingMask);
                serialized = true;
            }
        }

        FREETMPS;
        LEAVE;

        if (serialized) {
            return;
        }
    }

    if (isFlag(blessingMask, BlessingFlags::AllowBlessed)) {
        adapter->OnNull();
        return;
    }

    croak("Found blessed object, no TO_JSON function or incorrect flags in this case");
}

void WritePerlObject(SV* sv, NYT::TYson2JsonCallbacksAdapter* adapter, const i32 blessingMask, bool isOuter) {
    if (SvPOKp(sv)) {
        STRLEN len;
        char* pvPtr = SvPV(sv, len);
        TStringBuf value(pvPtr, len);
        adapter->OnString(value);
    } else if (SvNOKp(sv)) {
        adapter->OnDouble(SvNV(sv));
    } else if (SvIOK_UV(sv)) {
        adapter->OnUInteger(SvUV(sv));
    } else if (SvIOK(sv)) {
        adapter->OnInteger(SvIV(sv));
    } else if (SvROK(sv)) {
        sv = SvRV(sv);
        svtype svt = SvTYPE(sv);

        if (svt == SVt_PV) {
            STRLEN len;
            char* pvPtr = SvPV(sv, len);
            TStringBuf value(pvPtr, len);
            adapter->OnString(value);
        } else if (svt == SVt_PVAV) {
            adapter->OnOpenArray();
            AV* avPtr = (AV*) sv;
            for (ui32 i = 0; i < av_len(avPtr) + 1; ++i) {
                WritePerlObject(*av_fetch(avPtr, i, true), adapter, blessingMask);
            }
            adapter->OnCloseArray();
        } else if (svt == SVt_PVHV) {
            if (strcmp(sv_reftype(sv, true), "HASH")) {
                HandleNamedHash(sv, adapter, blessingMask);
            } else {
                adapter->OnOpenMap();
                HV* hvPtr = (HV*) sv;
                ui32 size = hv_iterinit(hvPtr);
                for (ui32 i = 0; i < size; ++i) {
                    HE* iter = hv_iternext(hvPtr);
                    SV* keySv = hv_iterkeysv(iter);
                    STRLEN len;
                    char* pvPtr = SvPV(keySv, len);
                    TStringBuf keyValue(pvPtr, len);
                    adapter->OnMapKey(keyValue);
                    WritePerlObject(hv_iterval(hvPtr, iter), adapter, blessingMask);
                }
                adapter->OnCloseMap();
            }
        } else if (svt == SVt_PVMG) {
            if (!strcmp(sv_reftype(sv, true), "JSON::PP::Boolean")) {
                adapter->OnBoolean(SvIV(sv));
            } else {
                HandleNamedHash(sv, adapter, blessingMask);
            }
        } else if (svt < SVt_PVAV) {
            int boolType = ref_bool_type(sv);

            if (boolType == 1 || boolType == 0) {
                adapter->OnBoolean(boolType);
            } else if (isFlag(blessingMask, BlessingFlags::AllowUnknown)) {
                adapter->OnNull();
            } else {
                croak("cannot encode reference to scalar '%s' unless the scalar is 0 or 1",
                        SvPV_nolen(sv_2mortal(newRV_inc(sv))));
            }
        } else if (svt == SVt_PVCV) {
            if (isFlag(blessingMask, BlessingFlags::AllowUnknown)) {
                adapter->OnNull();
            } else {
                croak("cannot encode reference to subroutine and no AllowUnknown flag setted");
            }
        } else {
            croak("encountered %s, but JSON can only represent references to arrays or hashes",
                       SvPV_nolen(sv_2mortal(newRV_inc(sv))));
        }
    } else if (!SvOK(sv)) {
        adapter->OnNull();
    }
    else {
        croak("encountered perl type (%s,0x%x) that JSON cannot handle, check your input data",
                   SvPV_nolen(sv), (unsigned int)SvFLAGS(sv));
    }
}

void GetJsonValue(SV* jsonSv, NJson::TJsonValue& jsonValue) {
    TMemoryInput jsonInput(SvPV_nolen(jsonSv), SvLEN(jsonSv));
    NJson::ReadJsonTree(&jsonInput, &jsonValue, true);
}

void GetJsonValueFromYson(SV* ysonSv, NJson::TJsonValue& json) {
    TMemoryInput ysonInput(SvPV(ysonSv, SvLEN(ysonSv)), SvLEN(ysonSv));
    NJson2Yson::DeserializeYsonAsJsonValue(&ysonInput, &json);
}

TString GetYsonFromJson(SV* input) {
    NJson::TJsonValue jsonValue;
    GetJsonValue(input, jsonValue);

    TString buf;
    TStringOutput ysonOutput(buf);
    NJson2Yson::SerializeJsonValueAsYson(jsonValue, &ysonOutput);
    return buf;
}

MODULE = yson PACKAGE = yson

SV*
ConvertBase64Request2Json(request)
    SV* request
    CODE:
        try {
            const auto converter = NAppHost::NConverter::TConverterFactory().Create("service_request");

            TString input(SvPV(request, SvLEN(request)));
            TString jsonRequest = converter->ConvertToJSON(input);

            RETVAL = newSVpvn(jsonRequest.c_str(), jsonRequest.length());
        } catch (...) {
            croak(CurrentExceptionMessage().data());
        }
    OUTPUT:
        RETVAL

SV*
ConvertJson2Yson(input)
    SV* input
    CODE:
        try {
            TString yson = GetYsonFromJson(input);
            RETVAL = newSVpvn(yson.c_str(), yson.length());
        } catch (...) {
            croak(CurrentExceptionMessage().data());
        }
    OUTPUT:
        RETVAL

SV*
ConvertYson2PerlObject(ysonSv, bool utf8 = true, bool utf8HashKeys = false)
    SV* ysonSv
    CODE:
        try {
            STRLEN len = SvLEN(ysonSv);
            TMemoryInput inputStream(SvPV(ysonSv, len), len);

            SV* result;
            THolder<IPerlObjectBuilder> consumer(nullptr);
            if (utf8) {
                if (utf8HashKeys) {
                    consumer.Reset(new NJson2Yson::TSkipAttributesProxy<TPerlObjectBuilder<TMakeSVUtf8, TMakeSVUtf8>>());
                } else {
                    consumer.Reset(new NJson2Yson::TSkipAttributesProxy<TPerlObjectBuilder<TMakeSVUtf8, TMakeSV>>());
                }
            } else {
                if (utf8HashKeys) {
                    consumer.Reset(new NJson2Yson::TSkipAttributesProxy<TPerlObjectBuilder<TMakeSV, TMakeSVUtf8>>());
                } else {
                    consumer.Reset(new NJson2Yson::TSkipAttributesProxy<TPerlObjectBuilder<TMakeSV, TMakeSV>>());
                }
            }

            NYson::TYsonParser ysonParser(consumer.Get(), &inputStream, ::NYson::EYsonType::Node);
            ysonParser.Parse();
            result = consumer->GetRoot();

            RETVAL = newRV_noinc(result);
        } catch (...) {
            croak(CurrentExceptionMessage().data());
        }
    OUTPUT:
        RETVAL

SV*
ConvertPerlObject2Yson(perlObject, convertBlessed = 1, allowBlessed = 1, allowUnknown = 1)
    SV* perlObject
    int convertBlessed
    int allowBlessed
    int allowUnknown
    CODE:
        try {
            TString buf;
            TStringOutput outputStream(buf);

            i32 mask = 0;
            if (convertBlessed) {
                setFlag(mask, BlessingFlags::ConvertBlessed);
            }
            if (allowBlessed) {
                setFlag(mask, BlessingFlags::AllowBlessed);
            }
            if (allowUnknown) {
                setFlag(mask, BlessingFlags::AllowUnknown);
            }

            NYson::TYsonWriter ysonWriter(&outputStream, NYson::EYsonFormat::Binary, ::NYson::EYsonType::Node, false);
            NYT::TYson2JsonCallbacksAdapter adapter(&ysonWriter);
            WritePerlObject(perlObject, &adapter, mask, true);
            RETVAL = newSVpvn(buf.c_str(), buf.length());
        } catch (...) {
            croak(CurrentExceptionMessage().data());
        }
    OUTPUT:
        RETVAL

SV*
CompareYsonYson(originalYsonSv, toCompareYsonSv)
    SV* originalYsonSv
    SV* toCompareYsonSv
    CODE:
        try {
            NJson::TJsonValue originalJson;
            GetJsonValueFromYson(originalYsonSv, originalJson);

            NJson::TJsonValue toCompareJson;
            GetJsonValueFromYson(toCompareYsonSv, toCompareJson);

            TString res = NJsonWriter::TBuf().WriteJsonValue(&originalJson, true).Str() == NJsonWriter::TBuf().WriteJsonValue(&toCompareJson, true).Str() ? "Ok" : "Fail";
            RETVAL = newSVpvn(res.c_str(), res.length());
        } catch (...) {
            croak(CurrentExceptionMessage().data());
        }
    OUTPUT:
        RETVAL

SV*
CompareYsonJson(originalYsonSv, toCompareJsonSv)
    SV* originalYsonSv
    SV* toCompareJsonSv
    CODE:
        try {
            NJson::TJsonValue originalJson;
            GetJsonValueFromYson(originalYsonSv, originalJson);

            NJson::TJsonValue toCompareJson;
            GetJsonValue(toCompareJsonSv, toCompareJson);

            TString res = NJsonWriter::TBuf().WriteJsonValue(&originalJson, true).Str() == NJsonWriter::TBuf().WriteJsonValue(&toCompareJson, true).Str() ? "Ok" : "Fail";
            RETVAL = newSVpvn(res.c_str(), res.length());
        } catch (...) {
            croak(CurrentExceptionMessage().data());
        }
    OUTPUT:
        RETVAL

SV*
CompareJsonJson(originalJsonSv, toCompareJsonSv)
    SV* originalJsonSv
    SV* toCompareJsonSv
    CODE:
        try {
            NJson::TJsonValue originalJson;
            GetJsonValue(originalJsonSv, originalJson);

            NJson::TJsonValue toCompareJson;
            GetJsonValue(toCompareJsonSv, toCompareJson);

            TString res = NJsonWriter::TBuf().WriteJsonValue(&originalJson, true).Str() == NJsonWriter::TBuf().WriteJsonValue(&toCompareJson, true).Str() ? "Ok" : "Fail";
            RETVAL = newSVpvn(res.c_str(), res.length());
        } catch (...) {
            croak(CurrentExceptionMessage().data());
        }
    OUTPUT:
        RETVAL
