#!/usr/bin/env perl

use strict;
use warnings;
use yson;
use Time::HiRes qw(gettimeofday tv_interval);
use JSON::PP;
use Data::Dumper;
use Test::Deep::NoTest;

$Data::Dumper::Sortkeys = 1;

sub show {
    my $name = $_[0];
    my $Qvalues = $_[1];
    my $values = $_[2];
    print($name . ": \n");
    foreach (@{$Qvalues}) {
        print("\t" . sprintf("% 4s", "Q" . int($_ * 100)) . "\t=\t" . sprintf("% 10.4f", @{$values}[int($_ * (scalar(@{$values}) - 1))]) . "\n");
    };
}

open(base64file, "<" . $ARGV[0]);

my @Q = (0.25, 0.5, 0.6, 0.7, 0.75, 0.8, 0.85, 0.9, 0.95, 0.97, 0.99, 1.0);
my @keys = ('READ_YSON', 'WRITE_YSON', 'READ_AND_WRITE_YSON', 'READ_JSON', 'WRITE_JSON', 'READ_AND_WRITE_JSON');
my %times = ();

foreach (@keys) {
    @{$times{$_}} = ();
}

while (<base64file>) {
    my $json = yson::ConvertBase64Request2Json($_);
    my $yson = yson::ConvertJson2Yson($json);

    #Read Yson --> Perl
    my $starttime = [gettimeofday()];
    my $perl = yson::ConvertYson2PerlObject($yson, 1);
    my $endtime = [gettimeofday()];
    my $readtime = tv_interval($starttime, $endtime);
    $readtime *= 1000;
    push(@{$times{'READ_YSON'}}, $readtime);

    #Write Perl --> Yson
    $starttime = [gettimeofday()];
    my $out_yson = yson::ConvertPerlObject2Yson($perl);
    $endtime = [gettimeofday()];
    my $writetime = tv_interval($starttime, $endtime);
    $writetime *= 1000;
    push(@{$times{'WRITE_YSON'}}, $writetime);

    push(@{$times{'READ_AND_WRITE_YSON'}}, $readtime + $writetime);

    #Read Json --> Perl
    $starttime = [gettimeofday()];
    my $perlfromjson = JSON::PP->new->utf8->decode($json);
    $endtime = [gettimeofday()];
    $readtime = tv_interval($starttime, $endtime);
    $readtime *= 1000;
    push(@{$times{'READ_JSON'}}, $readtime);

    #Write Perl --> Json
    $starttime = [gettimeofday()];
    my $out_json = JSON::PP->new->encode($perlfromjson);
    $endtime = [gettimeofday()];
    $writetime = tv_interval($starttime, $endtime);
    $writetime *= 1000;
    push(@{$times{'WRITE_JSON'}}, $writetime);

    push(@{$times{'READ_AND_WRITE_JSON'}}, $readtime + $writetime);

    print(sprintf("% 5d", scalar @{$times{'READ_YSON'}}), ' ');

    #Check convertation
    #Compare original Yson and Yson from Perl
    print(sprintf("% 5s", yson::CompareYsonYson($yson, $out_yson)), " ");
    #Compare Yson from Perl and original Json
    print(sprintf("% 5s", yson::CompareYsonJson($out_yson, $json)), " ");
    #Compare Dumper dump of Perl from Yson and Perl from Json
    print(sprintf("% 5s", eq_deeply($perl, $perlfromjson) ? "Ok" : "Fail"), " ");
    #Compare Yson (Yson->Perl->Yson) and Json (Json->Perl->Json)
    print(sprintf("% 5s", yson::CompareYsonJson($out_yson, $out_json)), " ");
    #Compare Yson (Yson->Perl->Yson) and Json (Json->Perl->Json)
    print(sprintf("% 5s", yson::CompareJsonJson($json, $out_json)), " ");
    print("\n");
}

foreach (keys %times) {
    @{$times{$_}} = sort {$a <=> $b} @{$times{$_}};
    show($_, \@Q, \@{$times{$_}});
}

0
