#include "factor_storage.h"

#include <library/cpp/pybind/typedesc.h>

#include <kernel/factor_storage/factor_storage.h>
#include <util/stream/str.h>
#include <library/cpp/string_utils/base64/base64.h>

struct TFactorStorageHolder {
    THolder<TFactorStorage> Storage;

    TFactorStorageHolder(TFactorStorage* storage)
        : Storage(storage)
    {
    }
};

class TFactorStorageTraits: public NPyBind::TPythonType<TFactorStorageHolder, TFactorStorage, TFactorStorageTraits> {
private:
    using TParent = NPyBind::TPythonType<TFactorStorageHolder, TFactorStorage, TFactorStorageTraits>;
    friend class NPyBind::TPythonType<TFactorStorageHolder, TFactorStorage, TFactorStorageTraits>;
    TFactorStorageTraits();

public:
    static TFactorStorage* GetObject(TFactorStorageHolder& holder) {
        return holder.Storage.Get();
    }

    static TFactorStorageHolder* DoInitObject(PyObject* args, PyObject* kwargs);
    //static TFactorStorageHolder *DoInitPureObject(const TVector<TString> &);
};

class TFactorStorageSlicesInfoCaller: public NPyBind::TBaseMethodCaller<TFactorStorage> {
public:
    bool CallMethod(PyObject*, TFactorStorage* self, PyObject*, PyObject*, PyObject*& res) const override {
        try {
            TString state;
            state = SerializeFactorBorders(*self);
            NPyBind::TPyObjectPtr result(NPyBind::BuildPyObject(state), true);
            res = result.RefGet();
            return true;
        } catch (const std::exception& ex) {
            PyErr_SetString(PyExc_RuntimeError, ex.what());
        } catch (...) {
            PyErr_SetString(PyExc_RuntimeError, "Can't get state: unknown exception");
        }
        res = nullptr;
        return true;
    }
};

class TFactorStorageGetFactorsStringCaller: public NPyBind::TBaseMethodCaller<TFactorStorage> {
public:
    bool CallMethod(PyObject*, TFactorStorage* self, PyObject* args, PyObject*, PyObject*& res) const override {
        try {
            TString sliceName;
            if (!NPyBind::ExtractArgs(args, sliceName)) {
                sliceName = "all";
            }
            EFactorSlice slice = EFactorSlice::ALL;
            FromString(sliceName, slice);
            TString state;
            TStringOutput out(state);
            TFactorView view = (*self)[slice];
            bool isbegin = true;
            for (const auto& factor : view) {
                out << (isbegin ? "" : "\t") << factor;
                isbegin = false;
            }
            NPyBind::TPyObjectPtr result(NPyBind::BuildPyObject(state), true);
            res = result.RefGet();
            return true;
        } catch (const std::exception& ex) {
            PyErr_SetString(PyExc_RuntimeError, ex.what());
        } catch (...) {
            PyErr_SetString(PyExc_RuntimeError, "Can't get state: unknown exception");
        }
        res = nullptr;
        return true;
    }
};

class TFactorStorageGetFactorsCaller: public NPyBind::TBaseMethodCaller<TFactorStorage> {
public:
    bool CallMethod(PyObject*, TFactorStorage* self, PyObject* args, PyObject*, PyObject*& res) const override {
        try {
            TString sliceName;
            if (!NPyBind::ExtractArgs(args, sliceName)) {
                sliceName = "all";
            }
            EFactorSlice slice = EFactorSlice::ALL;
            FromString(sliceName, slice);
            TString state;
            TStringOutput out(state);
            TFactorView view = (*self)[slice];
            TVector<double> factors(view.begin(), view.end());
            NPyBind::TPyObjectPtr result(NPyBind::BuildPyObject(factors), true);
            res = result.RefGet();
            return true;
        } catch (const std::exception& ex) {
            PyErr_SetString(PyExc_RuntimeError, ex.what());
        } catch (...) {
            PyErr_SetString(PyExc_RuntimeError, "Can't get state: unknown exception");
        }
        res = nullptr;
        return true;
    }
};

// Methods
TFactorStorageTraits::TFactorStorageTraits()
    : TParent("libfactorstorage.TFactorStorage", "TFactorStorage wrapper")
{
    AddCaller("DumpSlicesInfo", new TFactorStorageSlicesInfoCaller);
    AddCaller("GetFactorsString", new TFactorStorageGetFactorsStringCaller);
    AddCaller("GetFactors", new TFactorStorageGetFactorsCaller);
}

TFactorStorageHolder* TFactorStorageTraits::DoInitObject(PyObject* args, PyObject*) {
    try {
        TString b64;
        if (!NPyBind::ExtractArgs(args, b64))
            ythrow yexception() << "Can't create TFactorStorage: expecting string parameter";
        TString data;
        Base64Decode(b64, data);
        TStringInput inp_stream(data);
        NFactorSlices::TSlicesMetaInfo hostInfo;
        THolder<TFactorStorage> loaded = NFSSaveLoad::Deserialize(&inp_stream, hostInfo);
        return new TFactorStorageHolder(loaded.Release());
    } catch (const std::exception& ex) {
        PyErr_SetString(PyExc_RuntimeError, ex.what());
    } catch (...) {
        PyErr_SetString(PyExc_RuntimeError, "Can't create TFactorStorage: unknown exception");
    }
    return nullptr;
}

void DoInitLibFactorStorage() {
    NPyBind::TPyObjectPtr m = NPyBind::TModuleHolder::Instance().InitModule("libfactorstorage");
    TFactorStorageTraits::Instance().Register(m, "TFactorStorage");
}
