#include <Python.h>
#include <kernel/geo/utils.h>
#include <library/cpp/pybind/typedesc.h>
#include <util/generic/map.h>
#include <util/generic/vector.h>
#include <util/charset/wide.h>
#include <utility>


struct TRegionsDBHolder {
    THolder<TRegionsDB> RegionsDB;

    TRegionsDBHolder(TRegionsDB *db)
        : RegionsDB(db) {
    }
};

struct TResolverDBHolder {
    THolder<TIPREG4Or6> IPREG;

    TResolverDBHolder(TIPREG4Or6* ipreg)
        : IPREG(ipreg) {
    }
};

class TRegionsDBTraits : public NPyBind::TPythonType<TRegionsDBHolder, TRegionsDB, TRegionsDBTraits> {
private:
    typedef class NPyBind::TPythonType<TRegionsDBHolder, TRegionsDB, TRegionsDBTraits> TParent;
    friend class NPyBind::TPythonType<TRegionsDBHolder, TRegionsDB, TRegionsDBTraits>;
    TRegionsDBTraits();

public:
    static TRegionsDB *GetObject(const TRegionsDBHolder &holder) {
        return holder.RegionsDB.Get();
    }

    static TRegionsDBHolder *DoInitObject(PyObject *args, PyObject* kwargs);
};

class TResolverDBTraits : public NPyBind::TPythonType<TResolverDBHolder, TIPREG4Or6, TResolverDBTraits> {
private:
    typedef class NPyBind::TPythonType<TResolverDBHolder, TIPREG4Or6, TResolverDBTraits> TParent;
    friend class NPyBind::TPythonType<TResolverDBHolder, TIPREG4Or6, TResolverDBTraits>;
    TResolverDBTraits();

public:
    static TIPREG4Or6 *GetObject(const TResolverDBHolder& holder) {
        return holder.IPREG.Get();
    }

    static TResolverDBHolder *DoInitObject(PyObject* args, PyObject* kwargs);
};


struct TRegionsNameMapHolder {
    THolder< TMap< TString, TVector<TGeoRegion> > > NameMap;

    TRegionsNameMapHolder(TRegionsDB *db)
        : NameMap(new TMap< TString, TVector<TGeoRegion> >) {
        if (db == nullptr)
            return;
        for (TGeoRegionDFSVector::const_iterator it = db->GetDFSVector().begin(), end = db->GetDFSVector().end(); it != end; ++it)
            (*NameMap)[db->GetName(it->first)].push_back(it->first);
    }
};

class TRegionsNameMapTraits : public NPyBind::TPythonType<TRegionsNameMapHolder, TMap< TString, TVector<TGeoRegion> >, TRegionsNameMapTraits> {
private:
    typedef class NPyBind::TPythonType<TRegionsNameMapHolder, TMap< TString, TVector<TGeoRegion> >, TRegionsNameMapTraits> TParent;
    friend class NPyBind::TPythonType<TRegionsNameMapHolder, TMap< TString, TVector<TGeoRegion> >, TRegionsNameMapTraits>;
    TRegionsNameMapTraits();

public:
    static TMap< TString, TVector<TGeoRegion> > *GetObject(const TRegionsNameMapHolder &holder) {
        return holder.NameMap.Get();
    }
};


class TGetNameMap : public NPyBind::TBaseMethodCaller<TRegionsDB> {
public:
    bool CallMethod(PyObject *, TRegionsDB *self, PyObject *, PyObject *, PyObject *&res) const override {
        NPyBind::TPyObjectPtr r(TRegionsNameMapTraits::Instance().CreatePyObject(new TRegionsNameMapHolder(self)));
        res = r.RefGet();
        return true;
    }
};


class TIsSubRegionOf: public NPyBind::TBaseMethodCaller<TRegionsDB> {
public:
    bool CallMethod(PyObject*, TRegionsDB* self, PyObject* args, PyObject* /*kwargs*/, PyObject*& res) const override {
        TGeoRegion child, parent;
        if (!NPyBind::ExtractArgs(args, child, parent))
            ythrow yexception() << "wrong parameters for TRegionsDB::IsSubRegionOf()";
        res = NPyBind::BuildPyObject(IsFromParentRegion(*self, child, parent));
        return true;
    }
};


class TGetParentRegionOfType : public NPyBind::TBaseMethodCaller<TRegionsDB> {
public:
    bool CallMethod(PyObject*, TRegionsDB* self, PyObject* args, PyObject*, PyObject*& res) const override {
        TGeoRegion currentRegion;
        size_t type;
        if(!NPyBind::ExtractArgs(args, currentRegion, type))
            ythrow yexception() << "wrong parametrs for TRegionDB::TGetParentRegionOfType. It should be: region id and region type";
        EGeoRegionType neededType = (EGeoRegionType)type;
        EGeoRegionType currentType = self->GetType(currentRegion);
        while(currentType != neededType) {
            TGeoRegion parentRegion = self->GetParent(currentRegion);
            currentType = self->GetType(parentRegion);
            if(parentRegion == currentRegion || (currentType == GRT_OTHER /*type 0*/ && currentType != neededType )) {
                Py_INCREF(Py_None);
                res = Py_None;
                return true;
            }
            currentRegion = parentRegion;
        }
        res = NPyBind::BuildPyObject(currentRegion);
        return true;
    }
};


class TResolveRegionByIP: public NPyBind::TBaseMethodCaller<TIPREG4Or6> {
public:
    bool CallMethod(PyObject*, TIPREG4Or6* self, PyObject* args, PyObject* /*kwargs*/, PyObject*& res) const override {
        TString ipAddr;
        if (!NPyBind::ExtractArgs(args, ipAddr))
            ythrow yexception() << "wrong parameters for TIPResolver::Resolve()";
        res = NPyBind::BuildPyObject(self->ResolveRegion(Ip4Or6FromString(ipAddr.data())));
        return true;
    }
};


class TResolveRegionName : public NPyBind::TBaseMethodCaller< TMap< TString, TVector<TGeoRegion> > > {
public:
    bool CallMethod(PyObject *, TMap< TString, TVector<TGeoRegion> > *self, PyObject *args, PyObject *, PyObject *&res) const override {
        if (!PyTuple_Check(args) || PyTuple_Size(args) != 1)
            ythrow yexception() << "Can't resolve region name: it should be exactly one parameter - region name";
        PyObject *regObj = PyTuple_GET_ITEM(args, 0);
        TString regName;
        if (!NPyBind::FromPyObject(regObj, regName))
            ythrow yexception() << "Can't resolve region name: it should be exactly one parameter - region name";
        TVector<TGeoRegion> result;
        TMap< TString, TVector<TGeoRegion> >::const_iterator it = self->find(regName);
        if (it != self->end())
            result = it->second;
        res = NPyBind::BuildPyObject(result);
        return true;
    }
};

class TRegionNamesStartswith : public NPyBind::TBaseMethodCaller< TMap< TString, TVector<TGeoRegion> > > {
public:
    bool CallMethod(PyObject *, TMap< TString, TVector<TGeoRegion> > *self, PyObject *args, PyObject *, PyObject *&res) const override {
        if (!PyTuple_Check(args) || PyTuple_Size(args) != 1)
            ythrow yexception() << "Can't resolve region name: it should be exactly one parameter - region name";
        PyObject *regObj = PyTuple_GET_ITEM(args, 0);
        TString regName;
        if (!NPyBind::FromPyObject(regObj, regName))
            ythrow yexception() << "Can't resolve region name: it should be exactly one parameter - region name";
        regName = WideToUTF8(to_lower(UTF8ToWide(regName)));
        TVector< std::pair<TString, TGeoRegion> > result;
        for (TMap< TString, TVector<TGeoRegion> >::const_iterator it = self->begin(); it != self->end(); ++it) {
            TString currentRegion = WideToUTF8(to_lower(UTF8ToWide(it->first)));
            if (currentRegion.StartsWith(regName))
                for (size_t ind = 0; ind < it->second.size(); ++ind) {
                    result.emplace_back(it->first, it->second[ind]);
                }
        }
        res = NPyBind::BuildPyObject(result);
        return true;
    }
};


TRegionsDBTraits::TRegionsDBTraits()
    : TParent("libkernelgeo.TRegionsDB", "geobase resolver") {
    AddCaller("GetName", NPyBind::CreateConstMethodCaller<TRegionsDB>(&TRegionsDB::GetName));
    AddCaller("GetNameUilCase", NPyBind::CreateConstMethodCaller<TRegionsDB>(&TRegionsDB::GetNameUilCase));
    AddCaller("GetEnglishName", NPyBind::CreateConstMethodCaller<TRegionsDB>(&TRegionsDB::GetEnglishName));
    AddCaller("GetParent", NPyBind::CreateConstMethodCaller<TRegionsDB>(&TRegionsDB::GetParent));
    AddCaller("GetType", NPyBind::CreateConstMethodCaller<TRegionsDB>(&TRegionsDB::GetType));
    AddCaller("GetTimezone", NPyBind::CreateConstMethodCaller<TRegionsDB>(&TRegionsDB::GetTimezone));
    AddCaller("GetChildren", NPyBind::CreateConstMethodCaller<TRegionsDB>(&TRegionsDB::GetChildren));
    AddCaller("IsSubRegionOf", new TIsSubRegionOf);
    AddCaller("GetNameMap", new TGetNameMap);
    AddCaller("GetParentRegionOfType", new TGetParentRegionOfType);
}

TRegionsDBHolder *TRegionsDBTraits::DoInitObject(PyObject *args, PyObject*) {
    if (!PyTuple_Check(args) || PyTuple_Size(args) != 1)
        ythrow yexception() << "Can't create RegionsDB: it should be exactly one parameter - path libgeodata3.bin";
    PyObject *geoDataPathObj = PyTuple_GET_ITEM(args, 0);
    TString geoDataPath(NPyBind::FromPyObject<TString>(geoDataPathObj));
    return new TRegionsDBHolder(new TRegionsDB(geoDataPath));
}

TResolverDBHolder *TResolverDBTraits::DoInitObject(PyObject* args, PyObject* /*kwargs*/) {
    if (!PyTuple_Check(args) || PyTuple_Size(args) != 1)
        ythrow yexception() << "Can't create RegionsDB: it should be exactly one parameter - path libgeodata3.bin";
    PyObject *geoDataPathObj = PyTuple_GET_ITEM(args, 0);
    TString geoDataPath(NPyBind::FromPyObject<TString>(geoDataPathObj));
    TGeobaseLookup gbLookup(geoDataPath.data());
    return new TResolverDBHolder(new TIPREG4Or6(gbLookup));
}

TRegionsNameMapTraits::TRegionsNameMapTraits()
    : TParent("libkernelgeo.TRegionsNameMap", "geobase name resolver") {
    AddCaller("Resolve", new TResolveRegionName);
    AddCaller("RegionStartswith", new TRegionNamesStartswith);
}

TResolverDBTraits::TResolverDBTraits()
        : TParent("libkernelgeo.TIPResolver", "geobase ip->region resolver") {
    AddCaller("Resolve", new TResolveRegionByIP);
}

NPyBind::TPyObjectPtr DoInitLibKernelGeo3() {
    NPyBind::TPyObjectPtr m = NPyBind::TModuleHolder::Instance().InitModule("libkernelgeo");
    TRegionsDBTraits::Instance().Register(m, "TRegionsDB");
    TResolverDBTraits::Instance().Register(m, "TIPResolver");

    return m;
}

void DoInitLibKernelGeo() {
    DoInitLibKernelGeo3();
}
