#include "analyze_word.h"
#include "lemma_info.h"

#include "python_wrappers.h"

#include <util/system/defaults.h>

using namespace NPython;

PyDoc_STRVAR(DocStr_Module, "Module for lexical analysis of words.");

PyDoc_STRVAR(DocStr_AnalyzeWord, "AnalyzeWord(word, langs=None, optlangs=None, split=True, accept_translit=False)\n\
\n\
Returns the result of analysis of a 'word': a list of objects of type LemmaInfo.\n\
Word is searched in context of languages in 'langs' and 'optlangs' (if values are given).\n\
    'word' - must be an unicode or str (in cp1251 encoding) object;\n\
    'langs', 'optlangs' - must be a list or tuple of str objects (names of languages, ex.:'en', 'ru');\n\
    'split' - just split word by dashes and apostrophes (ex: True, False);\n\
    'accept_translit' - use translit (ex: True, False);\n\
\n\
    if 'langs' is empty or omitted: word analysis is occuring in almost all languages,\n\
    else: analysis is occuring in languages from 'langs',\n\
    order of languages affects to their priorities (more recent - more prior);\n\
    Languages from 'optlangs' are added with the least priority to these from 'langs'.\n");

static PyMethodDef ModuleFunctions[] = {
    {"AnalyzeWord", reinterpret_cast<PyCFunction>(AnalyzeWord), METH_VARARGS | METH_KEYWORDS, DocStr_AnalyzeWord},
    {nullptr, nullptr, 0, nullptr}
};

// THISDLLNAME, MODULE_NAME_STR - macro definitions from lemmer_info.h
#if PY_MAJOR_VERSION >= 3
    #define MODULE_INITIALIZER PyMODINIT_FUNC Y_CAT(PyInit_, THISDLLNAME)(void)
#else
    #define MODULE_INITIALIZER PyMODINIT_FUNC Y_CAT(init, THISDLLNAME)(void)
#endif

MODULE_INITIALIZER {
    try {
        #if PY_MAJOR_VERSION >= 3
            static struct PyModuleDef moduledef = {
                PyModuleDef_HEAD_INIT, MODULE_NAME_STR, DocStr_Module, -1, ModuleFunctions, NULL, NULL, NULL, NULL
            };
            PyObject* module = PyModule_Create(&moduledef);
        #else
            PyObject* module = Py_InitModule3(MODULE_NAME_STR, ModuleFunctions, DocStr_Module);
        #endif

        if (module == nullptr) {
            ythrow yexception() << "failed to init module";
        }

        Py_TYPE(&TLemmaInfo::Type) = &PyType_Type;
        if (PyType_Ready(&TLemmaInfo::Type) != 0) {
            ythrow yexception() << "failed to init module";
        }

        PyObject* lemmaInfoType = reinterpret_cast<PyObject*>(&TLemmaInfo::Type);
        IncRefCount(lemmaInfoType);
        if (PyModule_AddObject(module, TLemmaInfo::ClassName, lemmaInfoType) != 0) {
            SafeDecRefCount(lemmaInfoType);
            ythrow yexception() << "failed to init module";
        }

        TPyObjPtr languages;
        languages.Reset(GetAllLanguages());
        if (PyModule_AddObject(module, "LANGUAGES", languages.Get()) != 0) {
            ythrow yexception() << "failed to init module";
        }
        languages.Release();

        #if PY_MAJOR_VERSION >= 3
            return module;
        #endif
    } catch (const yexception& exc) {
        #if PY_MAJOR_VERSION >= 3
            return nullptr;
        #else
            Raise(exc);
        #endif
    }
}
