#include "python_wrappers.h"

#include "bytesobject.h"

namespace NPython {

void IncRefCount(PyObject* ptr) {
    if (ptr == nullptr)
        return;
    Py_INCREF(ptr);
}

void SafeDecRefCount(PyObject* ptr) {
    if (ptr == nullptr)
        return;
    Py_XDECREF(ptr);
}

TPyObjPtr None() {
    IncRefCount(Py_None);
    return TPyObjPtr(Py_None);
}

TPyObjPtr True() {
    IncRefCount(Py_True);
    return TPyObjPtr(Py_True);
}

TPyObjPtr False() {
    IncRefCount(Py_False);
    return TPyObjPtr(Py_False);
}

// Error generating.

void Raise(const yexception& exc) {
    PyErr_SetString(PyExc_Exception, exc.what());
}

// Collections, containers.

// List functions/methods.

bool IsList(const TPyObjPtr& object) {
    return PyList_Check(object.Get());
}

TPyObjPtr List(size_t objCount) {
    TPyObjPtr list(PyList_New(objCount));
    if (list.Get() == nullptr) {
        ythrow yexception() << "bad alloc";
    }
    return list;
}

void ListAppend(const TPyObjPtr& list, const TPyObjPtr& item) {
    if (!IsList(list)) {
        ythrow yexception() << "not a list";
    }

    if (PyList_Append(list.Get(), item.Get())) {
        ythrow yexception() << "can not append list element";
    }
}

// Tuple functions/methods.

bool IsTuple(const TPyObjPtr& object) {
    return PyTuple_Check(object.Get());
}

TPyObjPtr Tuple(size_t objCount) {
    TPyObjPtr tuple(PyTuple_New(objCount));
    if (tuple.Get() == nullptr) {
        ythrow yexception() << "bad alloc";
    }
    return tuple;
}

void SetTupleItem(const TPyObjPtr& tuple, size_t index, const TPyObjPtr& item) {
    if (!IsTuple(tuple)) {
        ythrow yexception() << "not a tuple";
    }
    PyTuple_SET_ITEM(tuple.Get(), index, item.Get());
    IncRefCount(item.Get());
}

TPyObjPtr GetItem(const TPyObjPtr& object, size_t index) {
    if (IsTuple(object)) {
        return TPyObjPtr(PyTuple_GetItem(object.Get(), index), true);
    } else if (IsList(object)) {
        return TPyObjPtr(PyList_GetItem(object.Get(), index), true);
    } else {
        ythrow yexception() << "unknown array type";
    }
}

size_t GetSize(const TPyObjPtr& object) {
    if (IsList(object)) {
        return PyList_Size(object.Get());
    } else if (IsTuple(object)) {
        return PyTuple_Size(object.Get());
    } else {
        ythrow yexception() << "unknown array type";
    }
}

TString Convert(const char* string) {
    return RecodeFromYandex(CODES_UTF8, string);
}

TString Convert(const TChar* string) {
    return Convert(TUtf16String(string));
}

TString Convert(const TString& string) {
    return string;
}

TString Convert(const TUtf16String& string) {
    return WideToUTF8(string.c_str(), string.size());
}

bool IsString(const TPyObjPtr& string) {
    return PyBytes_Check(string.Get());
}

bool IsUnicode(const TPyObjPtr& string) {
    return PyUnicode_Check(string.Get());
}

TPyObjPtr UnicodeAsUTF8String(const TPyObjPtr& string) {
    if (!IsUnicode(string)) {
        ythrow yexception() << "not an unicode string";
    }
    TPyObjPtr encoded(PyUnicode_AsUTF8String(string.Get()));
    if (encoded.Get() == nullptr) {
        ythrow yexception() << "encoding error";
    }
    return encoded;
}

TString CppString(PyObject* string) {
    const char* translated;
#if PY_MAJOR_VERSION >= 3
    if (PyUnicode_Check(string)) {
        translated = PyUnicode_AsUTF8(string);
    } else if (PyBytes_Check(string)) {
        translated = PyBytes_AsString(string);
    } else {
        ythrow yexception() << "type error";
    }
#else
    translated = PyBytes_AsString(string);
    if (translated == nullptr) {
        ythrow yexception() << "type error";
    }
#endif
    return translated;
}

TString CppString(const TPyObjPtr& string) {
    return CppString(string.Get());
}

} // namespace Python
