#pragma once

#include "python_core.h"

#include <util/generic/ptr.h>
#include <util/generic/yexception.h>
#include <util/generic/yexception.h>
#include <util/charset/wide.h>
#include <library/cpp/charset/recyr.hh>

namespace NPython {

// General.

void IncRefCount(PyObject* ptr);
void SafeDecRefCount(PyObject* ptr);

void Raise(const yexception& exc);

// Auto pointer template for derivatives of PyObject.
template <class Type>
class TDeletePyObject {
public:
    static void Destroy(Type* ptr) noexcept {
        SafeDecRefCount(reinterpret_cast<PyObject*>(ptr));
    }
};

template <class Type>
class TPythonPointer : public TSimpleSharedPtr<Type, TDeletePyObject<Type>> {
private:
    typedef TSimpleSharedPtr<Type, TDeletePyObject<Type>> TBase;

public:
    explicit TPythonPointer(Type* t = nullptr, bool incRef = false) noexcept
        : TBase(t)
    {
        if (incRef) {
            IncRefCount(reinterpret_cast<PyObject*>(t));
        }
    }

    Type* Release() {
        Type* res = this->Get();
        IncRefCount(reinterpret_cast<PyObject*>(res));

        this->Reset(TPythonPointer<Type>());

        return res;
    }
};

// Auto pointer for PyObject.
typedef TPythonPointer<PyObject> TPyObjPtr;

// Constructors for elementar python objects

TPyObjPtr None();
TPyObjPtr True();
TPyObjPtr False();

// List functions/methods.

bool IsList(const TPyObjPtr& object);
TPyObjPtr List(size_t objCount = 0);
size_t GetSize(const TPyObjPtr& list);
void ListAppend(const TPyObjPtr& list, const TPyObjPtr& item);

// Tuple functions/methods.

bool IsTuple(const TPyObjPtr& object);
TPyObjPtr Tuple(size_t objCount);
void SetTupleItem(const TPyObjPtr& tuple, size_t index, const TPyObjPtr& item);

// Common array's functions

size_t GetSize(const TPyObjPtr& object);
TPyObjPtr GetItem(const TPyObjPtr& object, size_t index);

// Unicode and string functions.

TString Convert(const char* string);
TString Convert(const TChar* string);
TString Convert(const TString& string);
TString Convert(const TUtf16String& string);

bool IsString(const TPyObjPtr& string);
bool IsUnicode(const TPyObjPtr& string);

TPyObjPtr UnicodeAsUTF8String(const TPyObjPtr& string);
TString CppString(PyObject* string);
TString CppString(const TPyObjPtr& string);

template <class T>
TPyObjPtr PythonString(T string) {
    TString temp = Convert(string);
    TPyObjPtr unicode(PyUnicode_DecodeUTF8(temp.c_str(), temp.length(), nullptr));
    if (unicode.Get() == nullptr) {
        ythrow yexception() << "bad alloc";
    }
    return unicode;
}

} // namespace NPython
