#include <library/cpp/pybind/typedesc.h>

#include <dict/dictutil/str.h>
#include <dict/misspell/errmodels/lib/errmodels.h>
#include <dict/misspell/errmodels/lib/transfeme/metric.h>
#include <dict/misspell/loader/corrector.h>
#include <dict/misspell/prompter/lib/normalize.h>
#include <library/cpp/charset/wide.h>
#include <library/cpp/langs/langs.h>
#include <util/generic/ptr.h>

using namespace NMisspell;

///////////////////////////////////////////////////////////////////////////////////////////////////
//
//  Helpers
//

class TInvalidArgException : public yexception {};

static PyObject* ExceptionClass(const std::exception* exc) {
    if (dynamic_cast<const TIoException*>(exc))
        return PyExc_IOError;
    if (dynamic_cast<const TInvalidArgException*>(exc))
        return PyExc_ValueError;
    return PyExc_RuntimeError;
}

static void TranslateCPPException(const std::exception& exc) {
    PyErr_SetString(ExceptionClass(&exc), WideToChar(CharToWide(exc.what(), CODES_YANDEX), CODES_ASCII).c_str());
}


///////////////////////////////////////////////////////////////////////////////////////////////////
//
//  String metrics wrappers
//

struct TSMetricHolder {
    ptr_to<TSMetric> SMetric;

    TSMetricHolder(TSMetric* sMetric, ERefType refType = REF_STRONG)
    :   SMetric(sMetric, refType) {}
};

class TEditDistanceTraits : public NPyBind::TPythonType<TSMetricHolder,
                                                        TSMetric,
                                                        TEditDistanceTraits>
{
private:
    typedef NPyBind::TPythonType<TSMetricHolder, TSMetric, TEditDistanceTraits> TBase;
    friend class NPyBind::TPythonType<TSMetricHolder, TSMetric, TEditDistanceTraits>;
    TEditDistanceTraits();

public:
    static TSMetric* GetObject(const TSMetricHolder& holder) {
        return holder.SMetric.get();
    }

    static TSMetricHolder* DoInitObject(PyObject* args, PyObject* kwargs);
};


class TTransfemeDistanceTraits : public NPyBind::TPythonType<TSMetricHolder,
                                                        TSMetric,
                                                        TTransfemeDistanceTraits>
{
private:
    typedef NPyBind::TPythonType<TSMetricHolder, TSMetric, TTransfemeDistanceTraits> TBase;
    friend class NPyBind::TPythonType<TSMetricHolder, TSMetric, TTransfemeDistanceTraits>;
    TTransfemeDistanceTraits();

public:
    static TSMetric* GetObject(const TSMetricHolder& holder) {
        return holder.SMetric.get();
    }

    static TSMetricHolder* DoInitObject(PyObject* args, PyObject* kwargs);
};


template<class TSubObject>
class TDistanceComputeCaller : public NPyBind::TBaseMethodCaller<TSubObject> {
public:
    bool CallMethod(PyObject*,
                            TSubObject* self,
                            PyObject* args, PyObject* /*kwargs*/, PyObject* &res) const override
    {
        try {
            if (self == nullptr) {
                throw yexception() << "String metric is poor initialized.";
            }

            TUtf16String from, to;
            if (!NPyBind::ExtractArgs(args, from, to)) {
                throw yexception() << "Can't parse arguments. Arguments should be unicode objects.";
            }

            std::unique_ptr<TSMetricRequest> request(self->NewRequest());
            request->Set(0, from);
            request->Set(1, to);
            request->Run();
            res = NPyBind::BuildPyObject(request->Distance());
            return true;
        } catch (const std::exception& ex) {
            PyErr_SetString(PyExc_RuntimeError, ex.what());
        } catch (...) {
            PyErr_SetString(PyExc_RuntimeError, "Unknown exception while checking");
        }
        res = nullptr;
        return true;
    }
};

TEditDistanceTraits::TEditDistanceTraits()
    : TBase(
        "EditDistance",
        "Returns edit distance-based metric enhanced by manual rules for specific languages."
    )
{
    AddCaller("Calculate", new TDistanceComputeCaller<TSMetric>());
}

TTransfemeDistanceTraits::TTransfemeDistanceTraits()
    : TBase(
        "TransfemeDistance",
        "Returns weighted edit distance with specified transfemes.")
{
    AddCaller("Calculate", new TDistanceComputeCaller<TSMetric>());
}

TSMetricHolder* TEditDistanceTraits::DoInitObject(PyObject* args, PyObject*) {
    try {
        ELanguage langID;
        if (args && PyTuple_Check(args) && PyTuple_Size(args) == 0) {
            langID = LANG_RUS;
        } else if (args && PyTuple_Check(args) && PyTuple_Size(args) == 1) {
            TString lang;
            if (!NPyBind::ExtractArgs(args, lang)) {
                throw yexception() << "Can't parse arguments.";
            }
            langID = LanguageByName(lang);
        } else {
            throw yexception() << "Can't parse arguments.";
        }
        auto metric = TSMetric::ForLang(langID);
        return new TSMetricHolder(const_cast<TSMetric*>(metric), REF_WEAK);
    } catch (const std::exception &ex) {
        PyErr_SetString(PyExc_RuntimeError, ex.what());
    } catch (...) {
        PyErr_SetString(PyExc_RuntimeError, "Unknown exception while checking");
    }
    return nullptr;
}

TSMetricHolder* TTransfemeDistanceTraits::DoInitObject(PyObject* args, PyObject*) {
    try {
        TString transfemePath;
        ELanguage langID;
        if (args && PyTuple_Check(args) && PyTuple_Size(args) == 1) {
            if (!NPyBind::ExtractArgs(args, transfemePath)) {
                throw yexception() << "Can't parse arguments.";
            }
            langID = LANG_RUS;
        } else if (args && PyTuple_Check(args) && PyTuple_Size(args) == 2) {
            TString lang;
            if (!NPyBind::ExtractArgs(args, transfemePath, lang)) {
                throw yexception() << "Can't parse arguments.";
            }
            langID = LanguageByName(lang);
        } else {
            throw yexception() << "Can't parse arguments.";
        }

        if (transfemePath.empty()) {
            throw yexception() << "Empty transfeme path.";
        }
        auto metric = CreateTransfemeMetric(transfemePath, CharsetForLang(langID));
        return new TSMetricHolder(metric.release()) ;
    } catch (const std::exception &ex) {
        PyErr_SetString(PyExc_RuntimeError, ex.what());
    } catch (...) {
        PyErr_SetString(PyExc_RuntimeError, "Unknown exception while checking");
    }
    return nullptr;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
//
//  IDocCorrector wrappers
//

class TDocCorrectorTraits : public NPyBind::TPythonType<IDocCorrector,
                                                        IDocCorrector,
                                                        TDocCorrectorTraits>
{
    typedef NPyBind::TPythonType<IDocCorrector, IDocCorrector, TDocCorrectorTraits> TBase;
    friend class NPyBind::TPythonType<IDocCorrector, IDocCorrector, TDocCorrectorTraits>;

private:
    TDocCorrectorTraits();
    class TCheck;

public:
    static IDocCorrector* GetObject(IDocCorrector& holder) {
        return &holder;
    }

    static IDocCorrector* DoInitObject(PyObject* args, PyObject* kwargs);
};

class TDocCorrectorTraits::TCheck : public NPyBind::TBaseMethodCaller<IDocCorrector> {
public:
    bool CallMethod(PyObject*,
                            IDocCorrector* self,
                            PyObject* args, PyObject* kwargs, PyObject* &res) const override
    {
        try {
            if (self == nullptr) {
                throw yexception() << "Poorly initialized misspell corrector";
            }
            TMisspellQuery query;
            static const char* kwd_names[] = {
                "text",
                "lang",
                nullptr
            };
            PyObject* textObj = nullptr;
            PyObject* langObj = nullptr;
            if (!PyArg_ParseTupleAndKeywords(args, kwargs, "U|S", (char**)kwd_names, &textObj, &langObj)) {
                res = nullptr;
                return true; // Parse raises exception itself
            }
            query.Text = NPyBind::FromPyObject<TUtf16String>(textObj);
            if (langObj)
                query.Lang = NPyBind::FromPyObject<TString>(langObj);
            else
                query.Lang = "";
            query.Options = 0;
            query.DebugLevel = 0;
            TMisspellResult result;
            Py_BEGIN_ALLOW_THREADS
            self->Check(query, result);
            Py_END_ALLOW_THREADS
            res = NPyBind::BuildPyObject(result.Text);
            return true;
        } catch (const std::exception& exc) {
            TranslateCPPException(exc);
        } catch (...) {
            PyErr_SetString(PyExc_RuntimeError, "Unknown exception while checking");
        }
        res = nullptr;
        return true;
    }
};

TDocCorrectorTraits::TDocCorrectorTraits()
    : TBase("DocCorrector", "Class binding to TDocCorrector from arcadia/dict/misspell")
{
    TBase::AddCaller("check", new TCheck);
}

IDocCorrector* TDocCorrectorTraits::DoInitObject(PyObject* args, PyObject*) {
    try {
        TString config;
        if (!NPyBind::ExtractArgs(args, config)) {
            throw TInvalidArgException() <<
                "Constructor for DocCorrector requires single argument - path to misspell.json";
        }
        THolder<IDocCorrector> DocCorrector(LoadCorrector(config).release());
        return DocCorrector.Release();
    } catch (const std::exception& exc) {
        TranslateCPPException(exc);
    } catch (...) {
        PyErr_SetString(PyExc_RuntimeError, "Unknown exception while checking");
    }
    return nullptr;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
//
//  dict/misspell/prompter/normalize wrapper
//

PyDoc_STRVAR(DocStr_Normalize, "Normalize(lang, text)\n\
\n\
Returns the result of normalization of a 'text';\n\
\n\
    lang must be 2 or 3 letters language code.\n");

PyObject* Normalize(PyObject*, PyObject* args, PyObject* kwargs) {
    static const char* kwd_names[] = {
                "lang",
                "text",
                nullptr
            };
    PyObject* result = nullptr;
    try {
        PyObject* textObj = nullptr;
        PyObject* langObj = nullptr;
        
        if (!PyArg_ParseTupleAndKeywords(args, kwargs, "S|U", (char**)kwd_names, &langObj, &textObj)) {
            return nullptr; // Parse raises exception itself
        }
        auto text = NPyBind::FromPyObject<TUtf16String>(textObj);
        auto lang = NPyBind::FromPyObject<TString>(langObj);
        ELanguage langCode = LanguageByNameOrDie(lang);
        auto normalized = NormalizeFuncForLanguage(langCode)(text);
        result = NPyBind::BuildPyObject(normalized);
    } catch (const std::exception& exc) {
        TranslateCPPException(exc);
    } catch (...) {
        PyErr_SetString(PyExc_RuntimeError, "Unknown exception while checking");
    }
    return result;
}

static PyMethodDef LibMisspellMethods[] = {
    {"Normalize", reinterpret_cast<PyCFunction>(Normalize), METH_VARARGS | METH_KEYWORDS, DocStr_Normalize},
    { nullptr, nullptr, 0, nullptr }
};

#if PY_MAJOR_VERSION >= 3
PyMODINIT_FUNC PyInit_yandex_misspell(void) {
    static struct PyModuleDef moduledef = {
        PyModuleDef_HEAD_INIT, "yandex_misspell", NULL, -1, LibMisspellMethods, NULL, NULL, NULL, NULL
    };
    PyObject* module = PyModule_Create(&moduledef);

    TDocCorrectorTraits::Instance().Register(module, "DocCorrector");
    TEditDistanceTraits::Instance().Register(module, "EditDistance");
    TTransfemeDistanceTraits::Instance().Register(module, "TransfemeDistance");

    return module;
}
#else
PyMODINIT_FUNC inityandex_misspell(void) {
    PyObject* module = Py_InitModule("yandex_misspell", LibMisspellMethods);
    TDocCorrectorTraits::Instance().Register(module, "DocCorrector");
    TEditDistanceTraits::Instance().Register(module, "EditDistance");
    TTransfemeDistanceTraits::Instance().Register(module, "TransfemeDistance");
}
#endif
