#!/usr/bin/python

import libperceptron
import math
import random


def SumSquareDiffs(x, y):
    se, max_diff = 0.0, 0.0
    for i in range(len(x)):
        se += ((x[i] - y[i]) * (x[i] - y[i]))
        max_diff = max(max_diff, abs(x[i] - y[i]))
    return se, max_diff

def CalcLogicalErrors(perceptron):
    rmse, max_diff = 0.0, 0.0
    for a in range(2):
        for b in range(2):
            res = perceptron.Calculate([float(a), float(b), 1.0])[-1]
            se, diff = SumSquareDiffs(res, [float(a ^ b), float(a & b), float(a | b), float(a), float(b)])
            rmse += se
            max_diff = max(max_diff, diff)
    return math.sqrt(rmse / 4), max_diff

def TestLogicalOperations():
    learner = libperceptron.TRMSELearner(0.1, [3, 4, 5])
    #print learner.Save()
    for s in range(100000):
        max_dw = 0.0
        for a in range(2):
            for b in range(2):
                learner.Add([float(a), float(b), 1.0], [float(a ^ b), float(a & b), float(a | b), float(a), float(b)])
                #print learner.GradientChecking([float(a), float(b), 1.0], [float(a ^ b), float(a & b), float(a | b), float(a), float(b)], 0.001)
        max_dw = max(max_dw, learner.FlushAdaDelta(0.95, 1e-1))
        #max_dw = max(max_dw, learner.FlushSimple(0.1))
        rmse, max_diff = CalcLogicalErrors(libperceptron.TRumelhartPerceptron(learner.Save()))
        if max_diff < 0.05:
            print "Quit on step: %d\nMaxDiff: %.4f\nRMSE: %.4f\nMaxDW: %.4f" % (s, max_diff, rmse, max_dw)
            break
    perceptron = libperceptron.TRumelhartPerceptron(learner.Save())
    for a in range(2):
        for b in range(2):
            res = perceptron.Calculate([float(a), float(b), 1.0])[-1]
            print "\t".join("%.4f" % t for t in res)
    for a in range(2):
        for b in range(2):
            res = perceptron.CalculateReduced([float(a), float(b), 1.0], -1)[-1]
            print "\t".join("%.4f" % t for t in res)


def GenerateBlenderSample(count):
    samples = []
    for i in range(count):
        rel = random.random() * 0.5
        web_rel = [0.6, 0.4, 0.25, 0.25, 0.18, 0.2, 0.15, 0.1, 0.05, 0.05]
        noise = 0.05
        p_win = lambda rel, i, rel_noise: math.exp(math.log(0.6) * i) * (rel - noise + 2 * rel_noise) if i != 10 else 0.0
        p_loss = lambda rel, i, rel_noise: 1.5 * math.exp(math.log(0.6) * (i + 1)) * (web_rel[i] - noise + 2 * rel_noise) if i != 10 else 0.0
        win_noises = [random.random() * noise for i in range(11)]
        loss_noises = [random.random() * noise for i in range(11)]
        scores = [(p_win(rel, i, win_noises[i]) - p_loss(rel, i, loss_noises[i])) for i in range(11)]
        features = [rel] + win_noises + loss_noises
        samples.append((features, scores))
    return samples

def CalcIdealSurplus(test):
    surplus = 0.0
    for item in test:
        surplus += max(item[1])
    return surplus

def CalcRandomSurplus(test):
    surplus = 0.0
    for item in test:
        surplus += item[1][int(random.random() * len(item[1]))]
    return surplus

def CalcPredictedSurplus(perceptron, test):
    surplus = 0.0
    err = 0.0
    eps = 0.00001
    for item in test:
        res = perceptron.Calculate(item[0])[-1]
        m = 0
        e, f = 0.0, 0.0
        for i in xrange(1, len(res)):
            e += (res[i] * item[1][i])
            f += res[i]
            if res[i] > res[m]:
                m = i
        surplus += item[1][m]
        err += (e / f)
    return surplus, err

def CalcMeanSigma(x):
    n = len(x)
    s, s2 = 0.0, 0.0
    for i in xrange(n):
        s += x[i]
        s2 += (x[i] * x[i])
    return s / n, math.sqrt(s2 / n / n - s * s / n / n / n)

def TestBlenderPacking():
    learn = GenerateBlenderSample(1000000)
    test = [GenerateBlenderSample(1000) for i in range(100)]
    results = [CalcIdealSurplus(t) for t in test]
    mean, sigma = CalcMeanSigma(results)
    print "Ideal:\t%.3f\t%.4f" % (mean, sigma)
    results = [CalcRandomSurplus(t) for t in test]
    mean, sigma = CalcMeanSigma(results)
    print "Random:\t%.3f\t%.4f" % (mean, sigma)
    learner = libperceptron.TArgmaxLearner(0.5, [len(learn[0][0]), 15, len(learn[0][1])])
    perceptron = libperceptron.TRumelhartPerceptron(learner.Save())
    results = [CalcPredictedSurplus(perceptron, t) for t in test]
    mean_surplus, sigma_surplus = CalcMeanSigma([r[0] for r in results])
    mean_err, sigma_err = CalcMeanSigma([r[1] for r in results])
    print "Initial:\t%.3f\t%.4f\t\t%.3f\t%.4f" % (mean_surplus, sigma_surplus, mean_err, sigma_err)
    for i in range(100):
        for item in learn:
            learner.Add(item[0], item[1])
        maxDw = learner.FlushAdaDelta(0.95, 1e-4)
        perceptron = libperceptron.TRumelhartPerceptron(learner.Save())
        results = [CalcPredictedSurplus(perceptron, t) for t in test]
        mean_surplus, sigma_surplus = CalcMeanSigma([r[0] for r in results])
        mean_err, sigma_err = CalcMeanSigma([r[1] for r in results])
        print "%.6f:\t%.3f\t%.4f\t\t%.3f\t%.4f" % (maxDw, mean_surplus, sigma_surplus, mean_err, sigma_err)

def GenerateArgmaxSample(count, a, b, n):
    samples = []
    for i in range(count):
        x = [a + random.random() * (b - a) for j in range(n)]
        samples.append((x, x))
    return samples

def TestArgmax():
    a, b, n = -1, 2, 10
    learn = GenerateArgmaxSample(100000, a, b, n)
    test = [GenerateArgmaxSample(1000, a, b, n) for i in range(100)]
    results = [CalcIdealSurplus(t) for t in test]
    mean, sigma = CalcMeanSigma(results)
    print "Ideal:\t%.3f\t%.4f" % (mean, sigma)
    results = [CalcRandomSurplus(t) for t in test]
    mean, sigma = CalcMeanSigma(results)
    print "Random:\t%.3f\t%.4f" % (mean, sigma)
    learner = libperceptron.TArgmaxLearner(1.0, [len(learn[0][0]), 10, len(learn[0][1])])
    perceptron = libperceptron.TRumelhartPerceptron(learner.Save())
    results = [CalcPredictedSurplus(perceptron, t) for t in test]
    mean_surplus, sigma_surplus = CalcMeanSigma([r[0] for r in results])
    mean_err, sigma_err = CalcMeanSigma([r[1] for r in results])
    print "Initial:\t%.3f\t%.4f\t\t%.3f\t%.4f" % (mean_surplus, sigma_surplus, mean_err, sigma_err)
    for i in range(10000):
        for item in learn:
            learner.Add(item[0], item[1])
        maxDw = learner.FlushAdaDelta(0.95, 1e-6)
        perceptron = libperceptron.TRumelhartPerceptron(learner.Save())
        results = [CalcPredictedSurplus(perceptron, t) for t in test]
        mean_surplus, sigma_surplus = CalcMeanSigma([r[0] for r in results])
        mean_err, sigma_err = CalcMeanSigma([r[1] for r in results])
        print "%.6f:\t%.3f\t%.4f\t\t%.3f\t%.4f" % (maxDw, mean_surplus, sigma_surplus, mean_err, sigma_err)

def RandomIntNumber(bit_count):
    number, bits, mult = 0, [], 1
    for i in range(bit_count):
        bit = 0 if random.random() < 0.5 else 1
        number += (bit * mult)
        bits.append(bit)
        mult *= 2
    return number, bits

def Bits2Number(bits, basis):
    number = 0
    mult = 1
    for i in range(len(bits)):
        bit = bits[i]
        if basis:
            mult = basis[i]
        number += (bit * mult)
        if not basis:
            mult *= 2
    return number

def Number2Bits(number, bit_count):
    bits = []
    for i in range(bit_count):
        bits.append(number % 2)
        number /= 2
    return bits

def GenerateSummatorSample(count, bit_count):
    result = []
    for i in range(count):
        a, a_bits = RandomIntNumber(bit_count)
        b, b_bits = RandomIntNumber(bit_count)
        result.append((a_bits + b_bits, a + b))
    return result

def CalcSums(perceptron, test, bit_count, basis, debug):
    result = 0.0
    for item in test:
        c_bits = perceptron.Calculate(item[0])[-1]
        a = Bits2Number(item[0][:bit_count], basis = None)
        b = Bits2Number(item[0][bit_count:], basis = None)
        c = Bits2Number(c_bits, basis = basis)
        if debug:
            print a, b, a + b, item[1], c
        result += ((a + b - c) * (a + b - c))
    return math.sqrt(result / len(test))

def TestSummator():
    bit_count = 10
    learn = GenerateSummatorSample(100000, bit_count)
    test = [GenerateSummatorSample(1000, bit_count) for i in range(100)]
    basis = [2 ** i for i in range(bit_count + 1)]
    learner = libperceptron.TRegressionLearner(basis, 1.0, [len(learn[0][0]), bit_count * bit_count, len(basis)])
    best, bestMean = None, 0.0
    n = 500
    for i in range(n):
        for item in learn:
            learner.Add(item[0], item[1])
        maxDw = learner.FlushAdaDelta(0.95, 1e-6)
        perceptron = libperceptron.TRumelhartPerceptron(learner.Save())
        results = [CalcSums(perceptron, t, bit_count, basis, False) for t in test]
        mean, sigma = CalcMeanSigma(results)
        if best == None or mean < bestMean:
            best = perceptron
            bestMean = mean
        print "Result: %.6f\t%.3f\t%.4f" % (maxDw, mean, sigma)
    print "Best: %.3f" % bestMean
    results = [CalcSums(best, t, bit_count, basis, True) for t in test]


def main():
    TestLogicalOperations()
    #TestBlenderPacking()
    #TestArgmax()
    #TestSummator()

if __name__ == "__main__":
    main()

