#include "captcha_check.h"

#include <library/cpp/charset/wide.h>
#include <library/cpp/charset/recyr.hh>
#include <library/cpp/unicode/utf8_iter/utf8_iter.h>
#include <util/charset/utf8.h>
#include <util/generic/vector.h>


static TString Translate(TStringBuf from, TStringBuf to, TStringBuf src) {
    TString result;
    for (auto c : TUtfIterBuf(src)) {
        bool translated = false;
        TUtfIterBuf fromIt(from), toIt(to);
        for (auto f : fromIt) {
            Y_ASSERT(toIt);
            auto t = toIt.GetNext();
            if (c == f) {
                result += t;
                translated = true;
                break;
            }
        }
        if (!translated) {
            Y_ASSERT(!toIt);
            result += c;
        }
    }
    return result;
}

TUtf16String CutBadSymbols(const TString& text) {
    TUtf16String result;
    for (auto ch : UTF8ToWide(text)) {
        if (IsAlphabetic(ch) || IsNumeric(ch)) {
            result += ch;
        }
    }
    return result;
}

static TString NormalizeAnswer(TStringBuf answer) {
    constexpr TStringBuf from = "АЕОЗЛЪЩЙЁЕÇçĞğIıİiÎîÖöLŞşÜüüOÂÛÊâûê";
    constexpr TStringBuf to   = "AE03ПЬШИEECCGG111111001SSUUU0AUEAUE";
    TUtf16String upper = TUtf16String::FromUtf8(answer);
    upper.to_upper();
    TString translated = Translate(from, to, WideToUTF8(upper));
    return WideToUTF8(CutBadSymbols(translated));
}

bool CheckSimpleAnswer(TStringBuf correctAnswer, TStringBuf userAnswer) {
    TVector<TString> userAnswers;

    try {
        userAnswers.push_back(NormalizeAnswer(userAnswer));
    } catch (...) {
    }

    try {
        userAnswers.push_back(NormalizeAnswer(Recode(CODES_WIN, CODES_UTF8, TString(userAnswer))));
    } catch (...) {
    }

    TString correctAnswerNormalized = NormalizeAnswer(correctAnswer);
    for (const auto& user : userAnswers) {
        if (user == correctAnswerNormalized) {
            return true;
        }
    }
    return false;
}

TMaybe<bool> CheckAnswer(TStringBuf correctAnswer, TStringBuf userAnswer) {
    if (!correctAnswer || !userAnswer) {
        return Nothing();
    }

    if (correctAnswer.find("#####_") != TString::npos) {
        return Nothing();
    }

    return CheckSimpleAnswer(correctAnswer, userAnswer);
}

TStringBuf TrimRep(const TStringBuf& rep, const TStringBuf& answer, bool usePrefix) {
    size_t min_len = std::min(rep.size(), answer.size());

    if (usePrefix) {
        return rep.Head(min_len);
    } else {
        return rep.Tail(rep.size() - min_len);
    }
}
