#!/usr/bin/env python

import argparse
import json
import logging
import os
import errno
import cStringIO
import tarfile
import time

from captcha.generation.image_generator import render, resources, textgen, parallel


def parse_args():
    parser = argparse.ArgumentParser(description='Generate captcha images')
    parser.add_argument('-c', '--config', required=True, type=argparse.FileType('r'), help='Path to config')
    parser.add_argument('-t', '--type', required=True, help='Captcha type to generate')
    parser.add_argument('-r', '--resources', required=True, type=argparse.FileType('r'), help='Path to resource archive')
    parser.add_argument('-f', '--format', choices=['png', 'gif'], default='gif', help='Output image format')
    parser.add_argument('-j', '--jobs', type=int, default=1, help='Number of concurrent jobs')
    parser.add_argument('-n', '--count', type=int, default=100, help='Number of images to generate')
    parser.add_argument('-m', '--max-retries-ratio', type=float, default=0.05, help='Max generate retry ratio (relative to count)')
    parser.add_argument('-v', '--verbose', action='store_true', help='Enable debug logging')

    subparsers = parser.add_subparsers(dest='output_type')

    parser_single_file = subparsers.add_parser('single_file', help='Generate only one image. Ignores --count')
    parser_single_file.add_argument('output_file', help='Path to output file')

    parser_debug_dir = subparsers.add_parser('debug_directory', help='Generate images in target directory and add file with metadata for debugging')
    parser_debug_dir.add_argument('target_dir', help='Path to target directory')

    parser_legacy_tarball = subparsers.add_parser('legacy_tarball', help='Generate tarball in legacy format (answer separated from content by null character)')
    parser_legacy_tarball.add_argument('output', type=argparse.FileType('w'), help='Path to tarball')
    parser_legacy_tarball.add_argument('--bare-answer', action='store_true', help='Write answers as bare text (like old implementation)')

    return parser.parse_args()


def write_image_callback(tag=None, text=None, image=None, image_format=None, target_dir=None):
    name = format(tag, '08') + '.' + image_format
    path = os.path.join(target_dir, name)
    image.save(path, format=image_format.upper())
    return (name, text)


def make_debug_directory(context, args):
    max_retries = int(args.max_retries_ratio * args.count)
    target_dir = args.target_dir

    try:
        os.makedirs(target_dir)
    except OSError as e:
        if e.errno != errno.EEXIST:
            raise

    kwargs = {
        'target_dir': target_dir,
        'image_format': args.format
    }
    answers = parallel.generate_parallel(context, args.count, args.jobs, write_image_callback, cb_kwargs=kwargs, max_retries=max_retries)
    answers_dict = dict(answers)
    answers_text = json.dumps(answers_dict, indent=4, sort_keys=True, ensure_ascii=False).encode('utf-8')
    with open(os.path.join(target_dir, 'metadata.json'), 'w') as metadata_file:
        metadata_file.write(answers_text)


def convert_to_legacy_format_callback(tag=None, text=None, image=None, image_format=None, target_dir=None, bare_answer=None):
    name = format(tag, '08') + '.e' + image_format
    output = cStringIO.StringIO()
    if bare_answer:
        header = text.encode('utf-8')
    else:
        content_type = 'image/%s; charset=utf-8' % image_format
        header = json.dumps({'answer': text, 'content_type': content_type})
    output.write(header)
    output.write('\0')
    image.save(output, format=image_format.upper())
    return name, output.getvalue()


def make_legacy_tarball(context, args):
    max_retries = int(args.max_retries_ratio * args.count)
    kwargs = {
        'bare_answer': args.bare_answer,
        'image_format': args.format
    }
    files = parallel.generate_parallel(context, args.count, args.jobs, convert_to_legacy_format_callback, cb_kwargs=kwargs, max_retries=max_retries)
    timestamp = time.time()
    with tarfile.open(fileobj=args.output, mode='w|gz') as tar:
        for name, content in files:
            name = args.type + '/' + name
            stream = cStringIO.StringIO(content)
            tarinfo = tarfile.TarInfo(name=name)
            tarinfo.size = len(content)
            tarinfo.mtime = timestamp
            tar.addfile(tarinfo=tarinfo, fileobj=stream)


def main():
    args = parse_args()

    if args.verbose:
        logging.getLogger().setLevel(logging.INFO)

    config = json.load(args.config)
    context = {
        'config': config[args.type],
        'resources': resources.Resources(args.resources)
    }

    if args.output_type == 'single_file':
        text = next(textgen.text_generator(context))
        img = render.render(context, text)
        img.save(args.output_file, format=args.format.upper())
    elif args.output_type == 'debug_directory':
        make_debug_directory(context, args)
    elif args.output_type == 'legacy_tarball':
        make_legacy_tarball(context, args)


if __name__ == '__main__':
    main()
