import multiprocessing
import logging

from captcha.generation.image_generator import textgen, render


def run_task(task, context):
    assert task['type'] == 'generate'
    tag = task['tag']
    text = task['text']
    callback = task['callback']
    cb_kwargs = task['callback_kwargs']
    try:
        img = render.render(context, text)
    except Exception as e:
        logging.error('Failed to generate text %s: %s' % (repr(text), e))
        return {
            'status': 'fail',
            'tag': tag
        }
    result = callback(tag=tag, text=text, image=img, **cb_kwargs)
    return {
        'status': 'success',
        'tag': tag,
        'result': result
    }


def worker_loop(context, task_queue, reply_queue):
    running = True
    while running:
        task = task_queue.get()
        if task['type'] == 'stop':
            running = False
        else:
            try:
                reply = run_task(task, context)
            except BaseException as e:
                reply = {
                    'tag': task['tag'],
                    'status': 'fatal',
                    'description': str(e)
                }
                running = False
            reply_queue.put(reply)


def generate_parallel(context, count, jobcount, callback, cb_kwargs={}, max_retries=0):
    task_queue = multiprocessing.Queue()
    reply_queue = multiprocessing.Queue()
    texts = textgen.text_generator(context)
    workers = [multiprocessing.Process(target=worker_loop, args=(context, task_queue, reply_queue)) for _ in xrange(jobcount)]

    for worker in workers:
        worker.start()

    results = [None]*count
    remaining = count
    retries = 0

    for i in xrange(count):
        task = {
            'type': 'generate',
            'tag': i,
            'text': next(texts),
            'callback': callback,
            'callback_kwargs': cb_kwargs
        }
        task_queue.put(task)

    try:
        while remaining:
            reply = reply_queue.get()
            if reply['status'] == 'success':
                results[reply['tag']] = reply['result']
                remaining -= 1
                logging.info('Remaining: %d' % remaining)
            elif reply['status'] == 'fail':
                retries += 1
                if retries > max_retries:
                    raise RuntimeError('Retries limit reached')

                task = {
                    'type': 'generate',
                    'tag': reply['tag'],
                    'text': next(texts),
                    'callback': callback,
                    'callback_kwargs': cb_kwargs
                }
                task_queue.put(task)
            elif reply['status'] == 'fatal':
                raise RuntimeError('Fatal error: %s' % reply['description'])
    finally:
        for i in xrange(jobcount):
            task_queue.put({'type': 'stop'})
        task_queue.close()
        for worker in workers:
            worker.join()

    return results
