# -*- coding: utf-8 -*-

import random


def build_validator_blacklist(conf, context):
    blacklist = context['resources'].get(conf['path']).decode('utf8').split()
    blacklist = [word.lower() for word in blacklist]

    def func(text):
        text = text.lower()
        for ss in blacklist:
            if ss in text:
                return False

        return True

    return func


def build_validator_forbidden_substrings(conf, context):
    substrings = conf['substrings']

    def func(text):
        for ss in substrings:
            if ss in text:
                return False

        return True

    return func


def build_validator(conf, context):
    vtype = conf['type']
    if vtype == 'blacklist':
        return build_validator_blacklist(conf, context)
    elif vtype == 'forbidden_substrings':
        return build_validator_forbidden_substrings(conf, context)
    else:
        raise RuntimeError("Unknown validator type %s" % vtype)


def build_validator_chain(context):
    configs = context['config']['generator'].get('validators', [])
    validators = [build_validator(conf, context) for conf in configs]

    def func(text):
        for v in validators:
            if not v(text):
                return False

        return True

    return func


def generate_text_legacy_rus():
    # Ported from Perl implementation
    RUS_LETTERS_M = u'АВЕКМН0РСТУХ'  # NOTE: '0' (digit) instead of cyrillic 'О', no idea why
    RUS_LETTER_LARGE = u'ШФДЖЮ'
    RUS_LETTER_NORMAL = u'ЦГЭЯИБ'

    wide_count = 0
    eng_count = 0
    result = u''
    for _ in xrange(6):
        letters = RUS_LETTER_NORMAL
        if wide_count <= 2:
            letters += RUS_LETTER_LARGE
        if eng_count <= 2:
            letters += RUS_LETTERS_M

        letter = random.choice(letters)
        if letter in RUS_LETTER_LARGE:
            wide_count += 1
        if letter in RUS_LETTERS_M:
            eng_count += 1

        result += letter

    return result


def legacy_g3_generator(conf):
    # Ported from Perl implementation
    family = conf.get('family', 'lat')
    letter_type = conf.get('letter_type', 'm')
    min_length = conf.get('length', {}).get('min', 6)
    max_length = conf.get('length', {}).get('max', 10)

    if not 6 <= min_length <= max_length:
        raise RuntimeError('Invalid min_length or max_length')

    letters = {
        'low': u'abcdefghijklmnopqrstuvwxyz',
        'low_wide': u'mw',
        'low_narrow': u'ilj',
        'up': u'ABCDEFGHJKLMNOPQRSTUVWXYZ',
        'digits': u'23456789'
    }

    type_av = {
        'l': ['low'],
        'u': ['up'],
        'm': ['low', 'up']
    }

    available = u''
    for ltype in type_av[letter_type]:
        available += letters[ltype]

    if family == 'mix':
        available += letters['digits']

    while True:
        length = random.randint(min_length, max_length)
        result = u''
        for _ in xrange(length):
            result += random.choice(available)

        if family == 'mix':
            dx = sum(c in letters['digits'] for c in result)
            if dx < 1:
                continue

        if letter_type in ('l', 'm'):
            lx = sum(c in letters['low'] for c in result)
            if lx < 3:
                continue

        if letter_type in ('u', 'm'):
            lx = sum(c in letters['up'] for c in result)
            if lx < 2:
                continue

        yield result


def generate_text_digits(conf):
    count = conf.get('digit_count', 6)
    result = u''
    for _ in xrange(count):
        result += unicode(random.randint(0, 9))

    return result


def apply_dictionary_legacy_uppercase(word):
    chance = .0
    if len(word) <= 4:
        chance = 0.6
    elif len(word) <= 5:
        chance = 0.5
    elif len(word) <= 6:
        chance = 0.35
    elif len(word) <= 7:
        chance = 0.1

    if random.random() < chance:
        return word.upper()
    else:
        return word


def dictionary_generator(context, conf):
    words = context['resources'].get(conf['path']).decode('utf8').split()
    while True:
        word = random.choice(words)
        if conf.get('legacy_uppercase'):
            word = apply_dictionary_legacy_uppercase(word)
        yield word


def unvalidated_text_generator(context):
    gen_config = context['config']['generator']
    gtype = gen_config['type']
    if gtype == 'legacy_rus':
        while True:
            yield generate_text_legacy_rus()
    if gtype == 'legacy_g3':
        for text in legacy_g3_generator(gen_config):
            yield text
    elif gtype == 'digits':
        while True:
            yield generate_text_digits(gen_config)
    elif gtype == 'dictionary':
        for text in dictionary_generator(context, gen_config):
            yield text
    else:
        raise RuntimeError("Unknown generator type %s" % gtype)


def text_generator(context):
    validator = build_validator_chain(context)
    for text in unvalidated_text_generator(context):
        if validator(text):
            yield text
