#!/usr/bin/env python
# -*- coding: utf-8 -*-

import yt.wrapper as yt
import argparse
from statface_client import StatfaceClient, StatfaceReportConfig


def parse_args():
    parser = argparse.ArgumentParser(description='Calculate OCR CAPTCHA input metrics')
    parser.add_argument('-c', '--yt-cluster', default='hahn', help='YT cluster.')
    parser.add_argument('-l', '--last-table', required=True, help='Last input table.')
    parser.add_argument('-a', '--all-results', required=True, help='Table with all input data.')
    parser.add_argument('-t', '--token-path', default='token', help='Path to the file with statface token.')
    return parser.parse_args()


def get_report_config():
    new_config = StatfaceReportConfig()
    config_in_yaml = u"""
    dimensions:
      - fielddate: date
    measures:
      - input_daily: number
      - input_all: number
    title: Количество картинок для разметки
    graphs:
    - type: line
      title: Количество присланных за день картинок для разметки
      fields:
      - input_daily
      titles:
        input_daily: Количество картинок
    - type: line
      title: Количество присланных за все время картинок для разметки
      fields:
      - input_all
      titles:
        input_all: Количество картинок
    """
    new_config.from_yaml(config_in_yaml)
    return new_config


def get_report(token_path):
    client_config = {
        'host': 'upload.stat.yandex-team.ru',
        'auth_config_path': token_path
    }
    client = StatfaceClient(client_config=client_config)
    report = client.get_report('Search_Spam/CAPTCHA/input_count')
    report.upload_config(get_report_config())
    return report


def main():
    args = parse_args()

    yt.config["proxy"]["url"] = args.yt_cluster

    input_daily = int(yt.row_count(args.last_table))
    input_all = int(yt.row_count(args.all_results))

    date = args.last_table.split("/")[-1]
    date = date.replace("_", "-")

    data = [{
        'fielddate': date,
        'input_daily': input_daily,
        'input_all': input_all,
    }]
    report = get_report(args.token_path)
    report.upload_data(scale='daily', data=data)


if __name__ == "__main__":
    main()
