#!/usr/bin/env python
# -*- coding: utf-8 -*-

import yt.wrapper as yt
import argparse
from statface_client import StatfaceClient, StatfaceReportConfig


def parse_args():
    parser = argparse.ArgumentParser(description='Calculate OCR CAPTCHA input with answers metrics')
    parser.add_argument('-c', '--yt-cluster', default='hahn', help='YT cluster.')
    parser.add_argument('-l', '--last-table', required=True, help='Last input with answers table.')
    parser.add_argument('-a', '--all-results', required=True, help='Table with all input with answers data.')
    parser.add_argument('-t', '--token-path', default='token', help='Path to the file with statface token.')
    return parser.parse_args()


def get_report_config():
    new_config = StatfaceReportConfig()
    config_in_yaml = u"""
    dimensions:
      - fielddate: date
    measures:
      - input_with_answers_weekly: number
      - input_with_answers_all: number
      - bad_images_ratio_weekly: number
    title: Количество картинок с известными ответами
    fill_missing_dates: 0
    graphs:
    - type: line
      title: Количество присланных за неделю картинок с ответами
      fields:
      - input_with_answers_weekly
      titles:
        input_with_answers_weekly: Количество картинок
    - type: line
      title: Количество присланных за все время картинок с ответами
      fields:
      - input_with_answers_all
      titles:
        input_with_answers_all: Количество картинок
    - type: line
      title: Доля плохих картинок среди присланных за неделю
      fields:
      - bad_images_ratio_weekly
      titles:
        bad_images_ratio_weekly: Доля плохих картинок
    view_types:
      bad_images_ratio_weekly:
        type: Float
        precision: 10
    """
    new_config.from_yaml(config_in_yaml)
    return new_config


def get_report(token_path):
    client_config = {
        'host': 'upload.stat.yandex-team.ru',
        "auth_config_path": token_path
    }
    client = StatfaceClient(client_config=client_config)
    report = client.get_report('Search_Spam/CAPTCHA/input_with_answers_count')
    report.upload_config(get_report_config())
    return report


def main():
    args = parse_args()

    yt.config["proxy"]["url"] = args.yt_cluster

    input_weekly = int(yt.row_count(args.last_table))
    ok_weekly = 0
    for record in yt.read_table(args.last_table):
        if record["Type"] == "OK" or record["Type"] == "DIFF":
            ok_weekly += 1
    bad_images_ratio = float(input_weekly - ok_weekly)/input_weekly

    input_all = int(yt.row_count(args.all_results))

    date = args.last_table.split("/")[-1]
    date = date.replace("_", "-")

    data = [{
        'fielddate': date,
        'input_with_answers_weekly': input_weekly,
        'input_with_answers_all': input_all,
        'bad_images_ratio_weekly': bad_images_ratio,
    }]
    report = get_report(args.token_path)
    report.upload_data(scale='daily', data=data)

if __name__ == "__main__":
    main()
