#!/usr/bin/env python
# -*- coding: utf-8 -*-

import yt.wrapper as yt
import argparse
from datetime import datetime
from statface_client import StatfaceClient, StatfaceReportConfig


def parse_args():
    parser = argparse.ArgumentParser(description='Calculate OCR CAPTCHA results metrics')
    parser.add_argument('-c', '--yt-cluster', default='hahn', help='YT cluster.')
    parser.add_argument('--date', required=True, help='File with date')
    parser.add_argument('--date-format', default="%Y-%m-%d", help='Date format')
    parser.add_argument('-l', '--last-table', required=True, help='Last results table.')
    parser.add_argument('-a', '--all-results', required=True, help='Table with all results.')
    parser.add_argument('-t', '--token-path', default='token', help='Path to the file with statface token.')
    return parser.parse_args()


def get_report_config():
    new_config = StatfaceReportConfig()
    config_in_yaml = u"""
    dimensions:
      - fielddate: date
    measures:
      - markuped_daily: number
      - markuped_all: number
      - markuped_all_target: number
    title: Количество размеченных картинок
    graphs:
    - type: line
      title: Количество размеченных за день картинок
      fields:
      - markuped_daily
      titles:
        markuped_daily: Количество картинок
    - type: line
      title: Количество размеченных за все время картинок
      fields:
      - markuped_all
      - markuped_all_target
      titles:
        markuped_all: Количество размеченных картинок
        markuped_all_target: Цель на Q2 2018
    """
    new_config.from_yaml(config_in_yaml)
    return new_config


def get_report(token_path):
    client_config = {
        'host': 'upload.stat.yandex-team.ru',
        "auth_config_path": token_path
    }
    client = StatfaceClient(client_config=client_config)
    report = client.get_report('Search_Spam/CAPTCHA/markuped_count')
    report.upload_config(get_report_config())
    return report


def main():
    args = parse_args()

    yt.config["proxy"]["url"] = args.yt_cluster

    markuped_daily = int(yt.row_count(args.last_table))
    markuped_all = int(yt.row_count(args.all_results))

    strdate = open(args.date, 'r').readline().strip()
    date = datetime.strptime(strdate, args.date_format)

    data = [{
        'fielddate': date.strftime('%Y-%m-%d'),
        'markuped_daily': markuped_daily,
        'markuped_all': markuped_all,
        'markuped_all_target': 3000000
    }]
    report = get_report(args.token_path)
    report.upload_data(scale='daily', data=data)

if __name__ == "__main__":
    main()
