#!/usr/bin/env python

from PIL import Image
from StringIO import StringIO
import yt.wrapper as yt
import argparse
import json
from captcha_tools import TarballContainer, NullContainer, generic_store_results

def enforce_image_size_limits(image, max_size):
    width, height = image.size
    max_width, max_height = max_size

    HORIZONTAL_SPACE_RATIO = 0.9
    max_width = int(max_width * HORIZONTAL_SPACE_RATIO)

    if width != max_width or height != max_height:
        if float(max_width) / max_height < float(width) / height:
            image = image.resize((max_width, int(float(height) * max_width / width)))
        else:
            image = image.resize((int(float(width) * max_height / height), max_height))

    return image

def pad_image(image, max_size):
    max_width, max_height = max_size
    width, height = image.size

    padded_image = Image.new('RGB', (max_width, max_height), (255, 255, 255))
    w_shift = int((max_width - width) / 2.0)
    h_shift = int((max_height - height) / 2.0)
    padded_image.paste(image, (w_shift, h_shift))
    return padded_image

def prepare_image(image_data, max_size):
    image = Image.open(StringIO(image_data))
    image = enforce_image_size_limits(image, max_size)
    image = pad_image(image, max_size)
    return image

def make_captcha_images(records_iter, max_size):
    max_width, max_height = max_size

    for record in records_iter:
        padded_unknown_image = prepare_image(record['unknown_image'], max_size)
        padded_known_image = prepare_image(record['known_image'], max_size)

        padded_result_image = Image.new('RGB', (2*max_width, max_height), (255, 255, 255))

        images = {
            'known': padded_known_image,
            'unknown': padded_unknown_image
        }

        left, right = record['layout'].split(',')

        padded_result_image.paste(images[left], (0, 0))
        padded_result_image.paste(images[right], (max_width, 0))

        del record['unknown_image']
        del record['known_image']
        record['image'] = padded_result_image
        yield record

def parse_args():
    parser = argparse.ArgumentParser(description='Prepare captcha images archive')

    parser.add_argument('--yt-proxy', default='hahn', help='YT proxy')

    parser.add_argument('--max-width', type=int, default=160, help='Max image width')
    parser.add_argument('--max-height', type=int, default=50, help='Max image height')

    parser.add_argument('--source-table', required=True, help='Path to the source table')

    parser.add_argument('--debug-tar', help='Gzipped tarball with images for debugging')
    parser.add_argument('--result-tar', help='Gzipped tarball with resulting images')

    parser.add_argument('--image-format', choices=['gif', 'png', 'jpeg'], default='gif', help='Image format')

    return parser.parse_args()

def store_results(image_iter, result_container, debug_container, image_format):
    def resultfunc(image_data):
        positions = {
            'known,unknown': 'left',
            'unknown,known': 'right'
        }
        metadata = {
            'known_position': positions[image_data['layout']],
            'known_answer': image_data['known_answer'],
            'unknown_id': image_data['unknown_id'].split('.')[0],
            'content_type': 'image/%s; charset=utf-8' % image_format
        }
        return image_data['image'], json.dumps(metadata)

    def indexfunc(image_data):
        return image_data['index']

    return generic_store_results(image_iter, result_container, debug_container, resultfunc, indexfunc=indexfunc, image_format=image_format)

def main():
    args = parse_args()

    yt.config['proxy']['url'] = args.yt_proxy
    max_size = args.max_width, args.max_height

    result_container = TarballContainer(args.result_tar)

    if args.debug_tar:
        debug_container = TarballContainer(args.debug_tar)
    else:
        debug_container = NullContainer()

    image_iter = make_captcha_images(yt.read_table(args.source_table), max_size)
    store_results(image_iter, result_container, debug_container, args.image_format)

if __name__ == '__main__':
    main()
