#include "caching_items_storage.h"

#include <library/cpp/logger/global/global.h>
#include <util/random/random.h>

namespace NCaptchaServer {
    TCaptchaCachingItemsStorage::TCaptchaCachingItemsStorage(const TCaptchaConfig& config, TCaptchaStats& stats, ICaptchaItemsStorage* slave)
        : Config(config)
        , Stats(stats)
        , Slave(slave)
        , UpdaterThread(UpdateLoop, this)
        , DataCache(Config.GetItemCache().GetItemsLimit())
    {
        UpdateIndex();

        auto metricCallback = [this]() -> double {
            TReadGuard rg(Mutex);
            return (Now() - LastUpdate).MilliSeconds();
        };
        Stats.RegisterSignalCallback(ESignals::TimeSinceLastIndexUpdateMs, metricCallback);

        UpdaterThread.Start();
    }

    NThreading::TFuture<bool> TCaptchaCachingItemsStorage::LoadItemData(const TCaptchaItemKey& key, TString& data) {
        with_lock (DataCacheMutex) {
            auto iter = DataCache.Find(key);
            if (iter != DataCache.End()) {
                data = *iter;
                Stats.PushSignal(ESignals::ItemCacheHits);
                return NThreading::MakeFuture(true);
            }
        }

        return Slave->LoadItemData(key, data).Apply([this, key, &data](const NThreading::TFuture<bool> fresult) {
            bool result = fresult.GetValue();
            if (result) {
                DEBUG_LOG << "Cache miss for key: " << key.DebugStr() << Endl;
                Stats.PushSignal(ESignals::ItemCacheMisses);

                with_lock (DataCacheMutex) {
                    DataCache.Update(key, data);
                }
                return true;
            } else {
                return false;
            }
        });
    }

    void TCaptchaCachingItemsStorage::LoadIndex(TCaptchaItemsIndex& index) {
        for (const auto& stored : *LastIndexes.GetCopy()) {
            index.insert(index.end(), stored.second.begin(), stored.second.end());
        }
    }

    TCaptchaItemHeader TCaptchaCachingItemsStorage::LoadRandomItemHeader(TStringBuf type, IRng* rng) {
        Y_ASSERT(rng);
        const auto& lastIndexesCopy = LastIndexes.GetCopy();
        const auto& index = lastIndexesCopy->at(type);
        return index[rng->RandomNumber(index.size())];
    }

    void TCaptchaCachingItemsStorage::UpdateIndex() {
        TCaptchaItemsIndex newIndex;
        Slave->LoadIndex(newIndex);

        TIndexesMap newLastIndexes;
        for (const auto& item : newIndex) {
            newLastIndexes[item.Key.Type].push_back(item);
        }
        LastIndexes.UpdateWithValue(newLastIndexes).Wait();

        TWriteGuard wg(Mutex);
        LastUpdate = Now();

        INFO_LOG << "Updated index cache" << Endl;
    }

    TCaptchaCachingItemsStorage::~TCaptchaCachingItemsStorage() {
        StopEvent.Signal();
        UpdaterThread.Join();
    }

    void* TCaptchaCachingItemsStorage::UpdateLoop(void* ptr) {
        TCaptchaCachingItemsStorage* thisptr = reinterpret_cast<TCaptchaCachingItemsStorage*>(ptr);
        while (true) {
            TDuration updateInterval = TDuration::Seconds(thisptr->Config.GetItemCache().GetUpdateIntervalSeconds());
            if (thisptr->StopEvent.WaitT(updateInterval)) {
                return nullptr;
            }

            try {
                thisptr->UpdateIndex();
            } catch (std::exception& ex) {
                ERROR_LOG << "Error in index cache update loop: " << ex.what() << Endl;
            } catch (...) {
                ERROR_LOG << "Error in index cache update loop (unknown type)" << Endl;
            }
        }
    }
}
