#include "captchatype.h"

#include <library/cpp/charset/wide.h>
#include <library/cpp/charset/recyr.hh>
#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/json/writer/json.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/html/pcdata/pcdata.h>
#include <library/cpp/unicode/utf8_iter/utf8_iter.h>

#include <util/charset/utf8.h>
#include <library/cpp/cgiparam/cgiparam.h>

namespace NCaptchaServer {
    void TCaptchaTypeSimple::InitSession(TStringBuf, int checks, IRng* rng, TCaptchaSessionInfo& sessInfo) {
        Y_ASSERT(rng);
        ICaptchaItemsStorage* itemsStorage = ItemsStorageRouter.GetStorage(sessInfo.ItemsStorage);
        TCaptchaItemHeader itemHeader = itemsStorage->LoadRandomItemHeader(ImageType, rng);
        sessInfo.Metadata["image_metadata"] = itemHeader.Metadata;
        sessInfo.Metadata["image_key"] = itemHeader.Key.AsJsonValue();

        if (checks > 0) {
            sessInfo.Checks = checks;
        }
    }

    bool TCaptchaTypeSimple::ResponseGenerate(TStringBuf, const TString& token, const TCaptchaSessionInfo& sessInfo, const TRequestInfo& reqInfo, THttpResponse& response) {
        TParsedHttpFull req(reqInfo.HttpInput.FirstLine());
        TCgiParameters params(req.Cgi);

        bool error = false;
        TStringBuf errorMessage;

        TString schema = "//";
        bool https = false;
        if (EqualToOneOf(params.Get("https"), "1", "on", "", "0")) {
            https = true;
            schema = "https://";
        } else if (params.Get("https") != "any") {
            schema = "http://";
        }

        bool voice = false;
        TString voiceUrl, voiceIntroUrl;

        if (params.Has("vtype")) {
            voice = true;
            voiceUrl = schema + Config.GetServiceUrl() + "/voice?key=" + token;
            voiceIntroUrl = schema + sessInfo.Metadata["voice_introurl"].GetStringSafe();
        }

        TString url = schema + Config.GetServiceUrl() + "/image?key=" + token;

        TString retpath = params.Get("retpath");

        int checks;
        if (!params.Get("checks")) {
            checks = 1;
        } else if (!TryFromString(params.Get("checks"), checks)) {
            checks = -1;
        }

        if (checks < 0 || checks > 10) {
            error = true;
            errorMessage = TStringBuf("checks must be less then 10");
        }

        if (!EqualToOneOf(params.Get("json"), "", "0")) {
            TString result;
            TStringOutput so(result);
            NJsonWriter::TBuf json(NJsonWriter::HEM_DONT_ESCAPE_HTML, &so);
            json.BeginObject();
            if (!error) {
                json.WriteKey("imageurl").WriteString(url, NJsonWriter::HEM_UNSAFE);
                if (https) {
                    json.WriteKey("https").WriteInt(1);
                }
                if (voice) {
                    json.WriteKey("voiceurl").WriteString(voiceUrl, NJsonWriter::HEM_UNSAFE);
                    json.WriteKey("voiceintrourl").WriteString(voiceIntroUrl, NJsonWriter::HEM_UNSAFE);
                }
                json.WriteKey("token").WriteString(token);
                if (retpath) {
                    json.WriteKey("retpath").WriteString(retpath, NJsonWriter::HEM_UNSAFE);
                }
            } else {
                json.WriteKey("error").WriteString(errorMessage);
            }
            json.WriteKey("json").WriteString("1");
            json.EndObject();

            response = THttpResponse(HTTP_OK);
            response.AddHeader("Content-Type", "application/json; charset=utf-8");
            response.SetContent(result);

            Stats.PushSignal(ESignals::TotalRequestsFormatJson);
        } else {
            TString result;
            TStringOutput so(result);
            so << "<?xml version=\"1.0\"?>\n";
            if (!error) {
                if (params.Get("style") == "old") {
                    so << "<number>" << token << "</number>";
                } else {
                    so << "<number url='" << url << "'";
                    TString attributes;
                    if (voice) {
                        so << " voiceurl='" << voiceUrl << "' voiceintrourl='" << voiceIntroUrl << "'";
                    }
                    if (retpath) {
                        so << " retpath='" << EncodeHtmlPcdata(retpath) << "'";
                    }
                    so << ">" << token << "</number>";
                }
            } else {
                so << "<error>" << errorMessage << "</error>";
            }

            response = THttpResponse(HTTP_OK);
            response.AddHeader("Content-Type", "text/xml; charset=utf-8");
            response.SetContent(result);

            Stats.PushSignal(ESignals::TotalRequestsFormatXml);
        }

        return !error;
    }

    NThreading::TFuture<THttpResponse> TCaptchaTypeSimple::HandleGetImage(TStringBuf, const TCaptchaSessionInfo& sessInfo, const TRequestInfo&) {
        TAtomicSharedPtr<TString> contentPtr(new TString());
        TCaptchaItemKey key(sessInfo.Metadata.GetMapSafe().at("image_key"));
        TString contentType = sessInfo.Metadata.GetMapSafe().at("image_metadata").GetMapSafe().at("content_type").GetStringSafe();

        auto cont = [contentPtr, contentType](const NThreading::TFuture<bool>& result) {
            if (!result.GetValue()) {
                return THttpResponse(HTTP_INTERNAL_SERVER_ERROR);
            }

            THttpResponse response(HTTP_OK);
            response.SetContent(*contentPtr);
            response.AddHeader("Content-Type", contentType);
            return response;
        };

        ICaptchaItemsStorage* itemsStorage = ItemsStorageRouter.GetStorage(sessInfo.ItemsStorage);
        return itemsStorage->LoadItemData(key, *contentPtr).Apply(cont);
    }

    static TString Translate(TStringBuf from, TStringBuf to, TStringBuf src) {
        TString result;
        for (auto c : TUtfIterBuf(src)) {
            bool translated = false;
            TUtfIterBuf fromIt(from), toIt(to);
            for (auto f : fromIt) {
                Y_ASSERT(toIt);
                auto t = toIt.GetNext();
                if (c == f) {
                    result += t;
                    translated = true;
                    break;
                }
            }
            if (!translated) {
                Y_ASSERT(!toIt);
                result += c;
            }
        }
        return result;
    }

    TUtf16String CutBadSymbols(const TString& text) {
        TUtf16String result;
        for (auto ch : UTF8ToWide(text)) {
            if (IsAlphabetic(ch) || IsNumeric(ch)) {
                result += ch;
            }
        }
        return result;
    }

    static TString NormalizeAnswer(TStringBuf answer) {
        constexpr TStringBuf from = "АЕОЗЛЪЩЙЁЕÇçĞğIıİiÎîÖöLŞşÜüüOÂÛÊâûê";
        constexpr TStringBuf to   = "AE03ПЬШИEECCGG111111001SSUUU0AUEAUE";
        TUtf16String upper = TUtf16String::FromUtf8(answer);
        upper.to_upper();
        TString translated = Translate(from, to, WideToUTF8(upper));
        return WideToUTF8(CutBadSymbols(translated));
    }

    bool CheckSimpleAnswer(TStringBuf correctAnswer, TStringBuf userAnswer) {
        TVector<TString> userAnswers;

        try {
            userAnswers.push_back(NormalizeAnswer(userAnswer));
        } catch (...) {
        }

        try {
            userAnswers.push_back(NormalizeAnswer(Recode(CODES_WIN, CODES_UTF8, TString(userAnswer))));
        } catch (...) {
        }

        TString correctAnswerNormalized = NormalizeAnswer(correctAnswer);
        for (const auto& user : userAnswers) {
            if (user == correctAnswerNormalized) {
                return true;
            }
        }
        return false;
    }

    bool TCaptchaTypeSimple::CheckAnswer(TStringBuf type, const TCaptchaSessionInfo& sessInfo, TStringBuf userAnswer) {
        Y_UNUSED(type);
        const auto& sessionMetadata = sessInfo.Metadata.GetMapSafe();
        return CheckSimpleAnswer(sessionMetadata.at("image_metadata").GetMapSafe().at("answer").GetStringSafe(), userAnswer);
    }

    bool TCaptchaTypeSimple::GetAnswer(TStringBuf, const TCaptchaSessionInfo& sessInfo, TString& answer) {
        const auto& sessionMetadata = sessInfo.Metadata.GetMapSafe();
        answer = sessionMetadata.at("image_metadata").GetMapSafe().at("answer").GetStringSafe();
        return true;
    }

    void TCaptchaTypeSimple::PrepareLoggedUserAnswer(TStringBuf, const TCaptchaSessionInfo&, TString& answer) {
        if (IsUtf(answer)) {
            return;
        }

        try {
            answer = Recode(CODES_WIN, CODES_UTF8, answer);
        } catch (...) {
            ythrow yexception() << "Invalid encoding: " << answer.Quote();
        }
    }
}
