#include "captchatype_localization.h"
#include "localization_checks.h"

#include <library/cpp/http/misc/parsed_request.h>

#include <library/cpp/json/json_reader.h>
#include <util/random/random.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <util/string/printf.h>
#include <util/string/split.h>

namespace NCaptchaServer {
    TCaptchaTypeLocalization::TCaptchaTypeLocalization(
        const TCaptchaConfig& config,
        TCaptchaStats& stats,
        TCaptchaItemsStorageRouter& itemsStorageRouter,
        const TString& knownImageType,
        const TString& unknownImageType,
        double requiredIOU,
        bool userCellsMustIntersectWithAnswer)

        : Config(config)
        , Stats(stats)
        , ItemsStorageRouter(itemsStorageRouter)
        , KnownImageType(knownImageType)
        , UnknownImageType(unknownImageType)
        , RequiredIOU(requiredIOU)
        , UserCellsMustIntersectWithAnswer(userCellsMustIntersectWithAnswer)
    {
    }

    static TRectangle MakeRandomBbox(IRng* rng, double size = 0.8) {
        Y_ASSERT(rng);
        const double bound = 1.0 - size;
        TRectangle result;

        result.MinX = round(rng->RandomDouble() * bound * 1000) / 1000.0;
        result.MinY = round(rng->RandomDouble() * bound * 1000) / 1000.0;
        result.MaxX = round((result.MinX + size) * 1000) / 1000.0;
        result.MaxY = round((result.MinY + size) * 1000) / 1000.0;

        return result;
    }

    void TCaptchaTypeLocalization::InitSession(TStringBuf, int checks, IRng* rng, TCaptchaSessionInfo& sessInfo) {
        Y_ASSERT(rng);
        ICaptchaItemsStorage* itemsStorage = ItemsStorageRouter.GetStorage(sessInfo.ItemsStorage);
        TCaptchaItemHeader knownImageHeader = itemsStorage->LoadRandomItemHeader(KnownImageType, rng);
        TCaptchaItemHeader unknownImageHeader = itemsStorage->LoadRandomItemHeader(UnknownImageType, rng);
        sessInfo.Metadata["known_image_metadata"] = knownImageHeader.Metadata;
        sessInfo.Metadata["known_image_key"] = knownImageHeader.Key.AsJsonValue();
        sessInfo.Metadata["known_image_bbox"] = MakeRandomBbox(rng).AsFlatString();
        sessInfo.Metadata["unknown_image_metadata"] = unknownImageHeader.Metadata;
        sessInfo.Metadata["unknown_image_key"] = unknownImageHeader.Key.AsJsonValue();
        sessInfo.Metadata["unknown_image_bbox"] = MakeRandomBbox(rng).AsFlatString();
        sessInfo.Metadata["known_index"] = rng->RandomNumber(2u);

        if (checks > 0) {
            sessInfo.Checks = checks;
        }
    }

    static void WriteImageInfo(NJsonWriter::TBuf& json, const NJson::TJsonValue& metadata, const TString& imageUrl, const TString& prefix) {
        const auto& metadataMap = metadata.GetMapSafe();
        json.BeginObject();
        json.WriteKey("url").WriteString(imageUrl, NJsonWriter::HEM_UNSAFE);
        json.WriteKey("category").WriteString(metadataMap.at(prefix + "_metadata").GetMapSafe().at("category_nominative").GetStringSafe());
        json.WriteKey("bbox").WriteString(metadataMap.at(prefix + "_bbox").GetStringSafe());
        json.WriteKey("grid").WriteString("4,4");
        json.EndObject();
    }

    bool TCaptchaTypeLocalization::ResponseGenerate(TStringBuf, const TString& token, const TCaptchaSessionInfo& sessInfo, const TRequestInfo& reqInfo, THttpResponse& response) {
        TParsedHttpFull req(reqInfo.HttpInput.FirstLine());
        TCgiParameters params(req.Cgi);

        TString schema = "//";
        bool https = false;
        if (EqualToOneOf(params.Get("https"), "1", "on", "", "0")) {
            https = true;
            schema = "https://";
        } else if (params.Get("https") != "any") {
            schema = "http://";
        }

        bool voice = false;
        TString voiceUrl, voiceIntroUrl;

        if (params.Has("vtype")) {
            voice = true;
            voiceUrl = schema + Config.GetServiceUrl() + "/voice?key=" + token;
            voiceIntroUrl = schema + sessInfo.Metadata["voice_introurl"].GetStringSafe();
        }

        TString result;
        TStringOutput so(result);
        NJsonWriter::TBuf json(NJsonWriter::HEM_DONT_ESCAPE_HTML, &so);
        json.BeginObject();

        json.WriteKey("images").BeginList();

        TString imageUrl1 = schema + Config.GetServiceUrl() + "/image?num=1&key=" + token;
        TString imageUrl2 = schema + Config.GetServiceUrl() + "/image?num=2&key=" + token;

        if (sessInfo.Metadata["known_index"] == 0) {
            WriteImageInfo(json, sessInfo.Metadata, imageUrl1, "known_image");
            WriteImageInfo(json, sessInfo.Metadata, imageUrl2, "unknown_image");
        } else {
            WriteImageInfo(json, sessInfo.Metadata, imageUrl1, "unknown_image");
            WriteImageInfo(json, sessInfo.Metadata, imageUrl2, "known_image");
        }

        json.EndList();
        if (https) {
            json.WriteKey("https").WriteInt(1);
        }
        if (voice) {
            json.WriteKey("voiceurl").WriteString(voiceUrl, NJsonWriter::HEM_UNSAFE);
            json.WriteKey("voiceintrourl").WriteString(voiceIntroUrl, NJsonWriter::HEM_UNSAFE);
        }
        json.WriteKey("token").WriteString(token);
        json.WriteKey("json").WriteString("1");
        json.EndObject();

        response = THttpResponse(HTTP_OK);
        response.AddHeader("Content-Type", "application/json; charset=utf-8");
        response.SetContent(result);

        Stats.PushSignal(ESignals::TotalRequestsFormatJson);
        return true;
    }

    NThreading::TFuture<THttpResponse> TCaptchaTypeLocalization::HandleGetImage(TStringBuf, const TCaptchaSessionInfo& sessInfo, const TRequestInfo& reqInfo) {
        TParsedHttpFull req(reqInfo.HttpInput.FirstLine());
        TCgiParameters params(req.Cgi);

        int num;
        if (!TryFromString(params.Get("num"), num) || num < 1 || num > 2) {
            return NThreading::MakeFuture(THttpResponse(HTTP_BAD_REQUEST));
        }

        TStringBuf prefix;
        const auto& metadataMap = sessInfo.Metadata.GetMapSafe();
        if (metadataMap.at("known_index") == 0) {
            prefix = (num == 1) ? TStringBuf("known") : TStringBuf("unknown");
        } else {
            prefix = (num == 1) ? TStringBuf("unknown") : TStringBuf("known");
        }

        TAtomicSharedPtr<TString> contentPtr(new TString());
        TCaptchaItemKey key(metadataMap.at(Sprintf("%s_image_key", prefix.data())));
        TString contentType = metadataMap.at(Sprintf("%s_image_metadata", prefix.data())).GetMapSafe().at("content_type").GetStringSafe();

        auto cont = [contentPtr, contentType](const NThreading::TFuture<bool>& result) {
            if (!result.GetValue()) {
                return THttpResponse(HTTP_INTERNAL_SERVER_ERROR);
            }

            THttpResponse response(HTTP_OK);
            response.SetContent(*contentPtr);
            response.AddHeader("Content-Type", contentType);
            return response;
        };
        ICaptchaItemsStorage* itemsStorage = ItemsStorageRouter.GetStorage(sessInfo.ItemsStorage);
        return itemsStorage->LoadItemData(key, *contentPtr).Apply(cont);
    }

    static TRectangle AnswerToRectangle(const NJson::TJsonValue& answer) {
        TRectangle result;

        const auto& minPoint = answer.GetArraySafe().at(0).GetArraySafe();
        const auto& maxPoint = answer.GetArraySafe().at(1).GetArraySafe();

        result.MinX = minPoint.at(0).GetDoubleSafe();
        result.MinY = minPoint.at(1).GetDoubleSafe();
        result.MaxX = maxPoint.at(0).GetDoubleSafe();
        result.MaxY = maxPoint.at(1).GetDoubleSafe();

        return result;
    }

    bool TCaptchaTypeLocalization::CheckAnswer(TStringBuf, const TCaptchaSessionInfo& sessInfo, TStringBuf userAnswer) {
        TVector<TString> responses;
        StringSplitter(userAnswer).Split(' ').AddTo(&responses);
        if (responses.size() != 2)
            return false;

        const auto& metadataMap = sessInfo.Metadata.GetMapSafe();
        const TString& validationResponse = responses.at(metadataMap.at("known_index").GetIntegerSafe());

        TResponseArray responseArray;
        for (const auto& it : StringSplitter(validationResponse).Split(',')) {
            int r;
            if (!TryFromString<int>(it.Token(), r))
                return false;

            responseArray.push_back(r);
        }

        const auto& answer = metadataMap.at("known_image_metadata").GetMapSafe().at("answer");
        TRectangle answerRect = AnswerToRectangle(answer);

        TRectangle bbox = TRectangle::FromFlatString(metadataMap.at("known_image_bbox").GetStringSafe());

        TLocalizationChecker checker(bbox, answerRect, responseArray, TGrid(4, 4));

        if (UserCellsMustIntersectWithAnswer && !checker.AllUserCellsIntersectWithAnswer()) {
            return false;
        }
        return checker.IntersectOverUnion() > RequiredIOU;
    }

    static void WriteImageAnswer(NJsonWriter::TBuf& json, const NJson::TJsonValue& metadata, const TString& prefix, bool hasAnswer) {
        const auto& metadataMap = metadata.GetMapSafe();
        const auto& imageMetadataMap = metadataMap.at(prefix + "_metadata").GetMapSafe();

        json.BeginObject();
        json.WriteKey("grid").BeginList();
        json.WriteString("4");
        json.WriteString("4");
        json.EndList();

        auto bbox = TRectangle::FromFlatString(metadataMap.at(prefix + "_bbox").GetStringSafe());
        json.WriteKey("bbox").BeginList();
        json.WriteString(ToString(bbox.MinX));
        json.WriteString(ToString(bbox.MinY));
        json.WriteString(ToString(bbox.MaxX));
        json.WriteString(ToString(bbox.MaxY));
        json.EndList();

        json.WriteKey("id").WriteString(imageMetadataMap.at("id").GetStringSafe());

        json.WriteKey("bbox_answer").BeginList();
        if (hasAnswer) {
            const auto& minPoint = imageMetadataMap.at("answer").GetArraySafe().at(0).GetArraySafe();
            const auto& maxPoint = imageMetadataMap.at("answer").GetArraySafe().at(1).GetArraySafe();
            json.WriteString(minPoint.at(0).GetStringRobust());
            json.WriteString(minPoint.at(1).GetStringRobust());
            json.WriteString(maxPoint.at(0).GetStringRobust());
            json.WriteString(maxPoint.at(1).GetStringRobust());
        }
        json.EndList();

        json.EndObject();
    }

    bool TCaptchaTypeLocalization::GetAnswer(TStringBuf, const TCaptchaSessionInfo& sessInfo, TString& answer) {
        TStringOutput so(answer);
        NJsonWriter::TBuf json(NJsonWriter::HEM_DONT_ESCAPE_HTML, &so);

        const auto& metadataMap = sessInfo.Metadata.GetMapSafe();
        json.BeginList();
        if (metadataMap.at("known_index").GetIntegerSafe() == 0) {
            WriteImageAnswer(json, sessInfo.Metadata, "known_image", true);
            WriteImageAnswer(json, sessInfo.Metadata, "unknown_image", false);
        } else {
            WriteImageAnswer(json, sessInfo.Metadata, "unknown_image", false);
            WriteImageAnswer(json, sessInfo.Metadata, "known_image", true);
        }
        json.EndList();

        return true;
    }

    static void WriteUserAnswer(NJsonWriter::TBuf& json, const TString& rep, const NJson::TJsonValue& metadata, const TString& prefix) {
        const auto& metadataMap = metadata.GetMapSafe();
        const auto& imageMetadataMap = metadataMap.at(prefix + "_metadata").GetMapSafe();

        json.BeginObject();
        json.WriteKey("cells").BeginList();
        for (const auto& s : StringSplitter(rep).Split(',')) {
            json.WriteString(s.Token());
        }
        json.EndList();
        json.WriteKey("id").WriteString(imageMetadataMap.at("id").GetStringSafe());
        json.EndObject();
    }

    void TCaptchaTypeLocalization::PrepareLoggedUserAnswer(TStringBuf, const TCaptchaSessionInfo& sessInfo, TString& userAnswer) {
        TString result;
        TStringOutput so(result);
        NJsonWriter::TBuf json(NJsonWriter::HEM_DONT_ESCAPE_HTML, &so);

        TVector<TString> responses;
        StringSplitter(userAnswer).Split(' ').AddTo(&responses);
        while (responses.size() < 2) {
            // not sure if this is meaningful, but this implements current behaviour
            responses.push_back("");
        }

        const auto& metadataMap = sessInfo.Metadata.GetMapSafe();
        json.BeginList();
        if (metadataMap.at("known_index").GetIntegerSafe() == 0) {
            WriteUserAnswer(json, responses.at(0), sessInfo.Metadata, "known_image");
            WriteUserAnswer(json, responses.at(1), sessInfo.Metadata, "unknown_image");
        } else {
            WriteUserAnswer(json, responses.at(0), sessInfo.Metadata, "unknown_image");
            WriteUserAnswer(json, responses.at(1), sessInfo.Metadata, "known_image");
        }
        json.EndList();

        userAnswer = result;
    }
}
