#include "captchatype_ocr.h"

#include <library/cpp/charset/recyr.hh>
#include <util/generic/algorithm.h>

namespace NCaptchaServer {
    template<typename TKnownStr, typename TAnswerStr>
    bool CheckOcrSubstr(const TKnownStr& knownAnswer, bool knownLeft, const TAnswerStr& userAnswer) {
        if (knownLeft) {
            return userAnswer.StartsWith(knownAnswer);
        } else {
            return userAnswer.EndsWith(knownAnswer);
        }
    }

    bool CheckOcrAnswer(const TString& knownAnswer, bool knownLeft, TStringBuf userAnswer) {
        TUtf16String knownAnswerWide, userAnswerWide;
        try {
            knownAnswerWide = CutBadSymbols(knownAnswer);
            userAnswerWide = CutBadSymbols(TString(userAnswer));
        } catch (...) {
            // use fallback for bad utf-8 strings
            return CheckOcrSubstr(knownAnswer, knownLeft, userAnswer);
        }
        return CheckOcrSubstr(knownAnswerWide, knownLeft, userAnswerWide);
    }

    bool TCaptchaTypeOcr::CheckAnswer(TStringBuf, const TCaptchaSessionInfo& sessInfo, TStringBuf userAnswer) {
        const auto& metadata = sessInfo.Metadata.GetMapSafe().at("image_metadata").GetMapSafe();
        const TString& knownAnswer = metadata.at("known_answer").GetStringSafe();
        const TString& knownPosition = metadata.at("known_position").GetStringSafe();
        bool knownLeft;

        knownLeft = (knownPosition == "left");
        if (!knownLeft && (knownPosition != "right")) {
            ythrow yexception() << "known_position has unexpected value " << knownPosition.Quote();
        }

        if (CheckOcrAnswer(knownAnswer, knownLeft, userAnswer)) {
            return true;
        }

        TString userAnswerFromCp1251;
        try {
            userAnswerFromCp1251 = Recode(CODES_WIN, CODES_UTF8, TString(userAnswer));
        } catch (...) {
            return false;
        }

        return CheckOcrAnswer(knownAnswer, knownLeft, userAnswerFromCp1251);
    }

    bool TCaptchaTypeOcr::GetAnswer(TStringBuf, const TCaptchaSessionInfo& sessInfo, TString& answer) {
        TStringOutput so(answer);
        const auto& sessionMetadata = sessInfo.Metadata.GetMapSafe();
        const auto& imageMetadata = sessionMetadata.at("image_metadata").GetMapSafe();
        const TString& knownAnswer = imageMetadata.at("known_answer").GetStringSafe();
        const TString& unknownId = imageMetadata.at("unknown_id").GetStringSafe();
        const TString& knownPosition = imageMetadata.at("known_position").GetStringSafe();

        if (knownPosition == "left") {
            so << knownAnswer << " #####_" << unknownId;
        } else if (knownPosition == "right") {
            so << "#####_" << unknownId << " " << knownAnswer;
        } else {
            ythrow yexception() << "known_position has unexpected value " << knownPosition.Quote();
        }

        return true;
    }
}
