#include <library/cpp/testing/unittest/registar.h>
#include <util/charset/wide.h>
#include "captchatype_ocr.h"

namespace NCaptchaServer {
    Y_UNIT_TEST_SUITE(CheckOcrAnswer) {
        Y_UNIT_TEST(Simple) {
            // knownAnswer  knownLeft  userAnswer
            UNIT_ASSERT(CheckOcrAnswer("cat", true, "cat ###unknown###"));
            UNIT_ASSERT(CheckOcrAnswer("cat", true, "cats ###unknown###"));
            UNIT_ASSERT(!CheckOcrAnswer("cat", true, "dog ###unknown###"));
            UNIT_ASSERT(!CheckOcrAnswer("cat", true, "###unknown### cat"));
            UNIT_ASSERT(!CheckOcrAnswer("cat", true, "###unknown### a cat"));

            UNIT_ASSERT(CheckOcrAnswer("cat", false, "###unknown### cat"));
            UNIT_ASSERT(CheckOcrAnswer("cat", false, "###unknown### a cat"));
            UNIT_ASSERT(!CheckOcrAnswer("cat", false, "###unknown### dog"));
            UNIT_ASSERT(!CheckOcrAnswer("cat", false, "cat ###unknown###"));
            UNIT_ASSERT(!CheckOcrAnswer("cat", false, "cats ###unknown###"));
        }

        Y_UNIT_TEST(SimpleUnicode) {
            UNIT_ASSERT(CheckOcrAnswer("кот", true, "кот ###unknown###"));
            UNIT_ASSERT(CheckOcrAnswer("кот", true, "коты ###unknown###"));
            UNIT_ASSERT(!CheckOcrAnswer("кот", true, "собака ###unknown###"));
            UNIT_ASSERT(!CheckOcrAnswer("кот", true, "###unknown### кот"));
            UNIT_ASSERT(!CheckOcrAnswer("кот", true, "###unknown### какой-то кот"));
            UNIT_ASSERT(!CheckOcrAnswer("кот", true, "кт ###unknown###"));

            UNIT_ASSERT(CheckOcrAnswer("кот", false, "###unknown### кот"));
            UNIT_ASSERT(CheckOcrAnswer("кот", false, "###unknown### какой-то кот"));
            UNIT_ASSERT(!CheckOcrAnswer("кот", false, "###unknown### собака"));
            UNIT_ASSERT(!CheckOcrAnswer("кот", false, "кот ###unknown###"));
            UNIT_ASSERT(!CheckOcrAnswer("кот", false, "коты ###unknown###"));
        }

        Y_UNIT_TEST(CutBadSymbols) {
            UNIT_ASSERT(CheckOcrAnswer("кот._", true, "кот__ собкак"));
            UNIT_ASSERT(CheckOcrAnswer("ко._!@?#$ %^&*()т", true, "к-_=+<>от собкак"));
            UNIT_ASSERT(CheckOcrAnswer("ко т", true, "ко.т  собкак"));
            UNIT_ASSERT(!CheckOcrAnswer("к0т", true, "кот      собкак"));
            UNIT_ASSERT(!CheckOcrAnswer("кот", true, "ко0т      собкак")); // check digit
        }

        Y_UNIT_TEST(Chinese) {
            UNIT_ASSERT(CheckOcrAnswer("世界你好", false, "00 世界你好"));
            // китайские символы вырезаются все
            UNIT_ASSERT(CheckOcrAnswer("1世界你好2", false, "12"));
        }

        Y_UNIT_TEST(CutBadSymbols2) {
            UNIT_ASSERT(CheckOcrAnswer("кіт", false, "00 кіт"));
            UNIT_ASSERT(!CheckOcrAnswer("кіт", false, "00 кіти"));
            UNIT_ASSERT(!CheckOcrAnswer("синій кіт", false, "00 кіт"));
            UNIT_ASSERT(!CheckOcrAnswer("кіт", false, "00 кт"));
        }

        Y_UNIT_TEST(CornerCases) {
            UNIT_ASSERT(!CheckOcrAnswer("кот", true, ""));
            UNIT_ASSERT(!CheckOcrAnswer("собкака", true, "SSSSSSSSAAAAAAAA)9"));
            UNIT_ASSERT(CheckOcrAnswer("", true, "SSSSSSSSAAAAAAAA)9"));
            UNIT_ASSERT(CheckOcrAnswer("))))", true, "any answer will pass"));
        }

        Y_UNIT_TEST(BadUtf1) {
            TString badStr;
            for (int i = 0; i < 256; i++) {
                badStr += i;
            }
            UNIT_ASSERT(!CheckOcrAnswer("answer", true, badStr));
        }

        Y_UNIT_TEST(BadUtf2) {
            TString badStr;
            for (int i = -10; i < 0; i++) {
                badStr += i;
            }
            UNIT_ASSERT(!CheckOcrAnswer("answer", true, badStr));

            badStr += " answer";
            UNIT_ASSERT(CheckOcrAnswer("answer", false, badStr)); // проверка fallback

            badStr += "*";
            UNIT_ASSERT(!CheckOcrAnswer("answer", false, badStr)); // проверка, что именно fallback сработал (спецсимвол не вырезан)
        }
    }
}
