#include "fallback_items_storage.h"

#include <contrib/libs/libarchive/libarchive/archive.h>
#include <contrib/libs/libarchive/libarchive/archive_entry.h>

#include <library/cpp/json/json_reader.h>

#include <util/generic/buffer.h>

namespace {
    class TArchiveDeleter {
    public:
        static inline void Destroy(struct archive* arc) noexcept {
            if (arc) {
                auto status = archive_read_free(arc);
                Y_VERIFY(status == ARCHIVE_OK);
            }
        }
    };

    void LoadArchive(TStringBuf path, THashMap<TString, TString>& contents) {
        THolder<struct archive, TArchiveDeleter> arc(archive_read_new());
        const size_t blockSize = 1024 * 10;

        Y_ENSURE(archive_read_support_format_all(arc.Get()) == ARCHIVE_OK);
        Y_ENSURE(archive_read_support_filter_all(arc.Get()) == ARCHIVE_OK);

        if (archive_read_open_filename(arc.Get(), path.data(), blockSize) != ARCHIVE_OK)
            ythrow yexception() << "Can't open archive " << TString{path}.Quote() << ": " << archive_error_string(arc.Get());

        while (true) {
            struct archive_entry* entry;
            int ret_code = archive_read_next_header(arc.Get(), &entry);

            if (ret_code == ARCHIVE_EOF)
                break;

            if (ret_code != ARCHIVE_OK)
                Y_ENSURE(ret_code == ARCHIVE_OK, "Can't read next entry: " << archive_error_string(arc.Get()));
            Y_ENSURE(S_ISREG(archive_entry_mode(entry)));

            const char* name = archive_entry_pathname(entry);
            Y_ENSURE(name != nullptr, "Can't get entry name: " << archive_error_string(arc.Get()));

            TBuffer body;
            size_t size = static_cast<size_t>(archive_entry_size(entry));
            if (size > 0) {
                body.Resize(size);
                if (archive_read_data(arc.Get(), body.Data(), size) != static_cast<ssize_t>(size)) {
                    ythrow yexception() << "Error while reading archive: " << archive_error_string(arc.Get());
                }
            }

            TString content;
            body.AsString(content);
            contents[name] = content;
        }
    }
}

namespace NCaptchaServer {
    TCaptchaFallbackItemsStorage::TCaptchaFallbackItemsStorage(const TCaptchaConfig& config, TCaptchaStats& stats)
        : Config(config)
        , Stats(stats)
    {
        Y_UNUSED(Stats);
        THashMap<TString, TString> archive;
        LoadArchive(config.GetFallback().GetItemsArchivePath(), archive);
        PrepareItems(archive);
    }

    NThreading::TFuture<bool> TCaptchaFallbackItemsStorage::LoadItemData(const TCaptchaItemKey& key, TString& data) {
        auto byType = Items.find(key.Type);
        if (byType == Items.end()) {
            return NThreading::MakeFuture(false);
        }
        auto item = byType->second.find(key);
        if (item == byType->second.end()) {
            return NThreading::MakeFuture(false);
        }

        data = item->second.Data;
        return NThreading::MakeFuture(true);
    }

    void TCaptchaFallbackItemsStorage::LoadIndex(TCaptchaItemsIndex& index) {
        index.clear();
        for (const auto& typeiter : Items) {
            for (const auto& iter : typeiter.second) {
                index.push_back(TCaptchaItemHeader(iter.first, iter.second.Metadata));
            }
        }
    }

    void TCaptchaFallbackItemsStorage::LoadIndex(TStringBuf type, TCaptchaItemsIndex& index) {
        index.clear();
        if (!Items.contains(type)) {
            ythrow yexception() << "Has no fallback item with type='" << type
                                << "'. Probably you need to regenerate fallback archive";
        }
        for (const auto& iter : Items.at(type)) {
            index.push_back(TCaptchaItemHeader(iter.first, iter.second.Metadata));
        }
    }

    void TCaptchaFallbackItemsStorage::PrepareItems(const THashMap<TString, TString>& archive) {
        NJson::TJsonValue index;
        ReadJsonTree(archive.at("index"), &index, true);
        for (const auto& item : index.GetMapSafe()) {
            const auto& data = archive.at(item.first);
            const auto& info = item.second.GetMapSafe();

            TCaptchaItemKey key;
            key.Type = info.at("type").GetStringSafe();
            key.Version = info.at("version").GetStringSafe();
            key.Id = info.at("id").GetStringSafe();

            Items[key.Type].emplace(key, TItem(data, info.at("metadata")));
        }
    }
}
