#include "fallback_session_storage.h"
#include "helpers.h"

#include <library/cpp/logger/global/global.h>

namespace NCaptchaServer {
    char TCaptchaFallbackSessionStorage::FallbackTokenTag = '9';

    TCaptchaFallbackSessionStorage::TCaptchaFallbackSessionStorage(const TCaptchaConfig& config, TCaptchaStats& stats, TCaptchaFallbackItemsStorage* fallbackItems)
        : Config(config)
        , Stats(stats)
        , FallbackItemsStorage(fallbackItems)
        , RotateThread(RotateLoop, this)
    {
        RandomTokenPartLength = Config.GetFallback().GetRandomTokenPartLength();
        Y_ENSURE(TokenLength > RandomTokenPartLength);

        SessionsPast = MakeHolder<TSessionsContainer>();
        SessionsPresent = MakeHolder<TSessionsContainer>();
        SessionsFuture = MakeHolder<TSessionsContainer>();

        TimeInterval = Config.GetSessionTimeoutSeconds();
        TimeCurrent = (Now().Seconds() / TimeInterval) * TimeInterval;
        ui64 timeSeed = TimeCurrent + Config.GetFallback().GetSeed();

        PrepareSessions(timeSeed - TimeInterval, *SessionsPast);
        PrepareSessions(timeSeed, *SessionsPresent);
        PrepareSessions(timeSeed + TimeInterval, *SessionsFuture);

        DEBUG_LOG << "Prepared fallback sessions (TimeCurrent=" << TimeCurrent << "; timeSeed=" << timeSeed << ")" << Endl;

        FillTypeAliases(config.GetCaptchaTypes(), CaptchaTypeAliases);
        FillTypeAliases(config.GetVoiceCaptchaTypes(), VoiceCaptchaTypeAliases);

        RotateThread.Start();
    }

    NThreading::TFuture<TString> TCaptchaFallbackSessionStorage::CreateSession(const TCaptchaSessionRequest& request, TCaptchaSessionInfo& info) {
        TStringBuf ctypeNormal = GetWithDefault(CaptchaTypeAliases, request.Type);
        TStringBuf vtypeNormal = GetWithDefault(VoiceCaptchaTypeAliases, request.VoiceType);
        TString token;

        {
            TReadGuard rg(Mutex);
            const auto& tokenPrefixes = SessionsPresent->TokenPrefixesByTypeCombos.at(std::make_pair(ctypeNormal, vtypeNormal));
            const auto& tokenPrefix = tokenPrefixes.at(RandomNumber(tokenPrefixes.size()));
            info = SessionsPresent->PreparedSessions.at(tokenPrefix);
            token = tokenPrefix;
        }

        TDefaultRng rng;
        TStringOutput so(token);
        RandomAlNumString(&rng, RandomTokenPartLength, so);
        Y_ASSERT(token.size() == TokenLength);
        return NThreading::MakeFuture(token);
    }

    NThreading::TFuture<bool> TCaptchaFallbackSessionStorage::LoadSessionInfo(TStringBuf token, TCaptchaSessionInfo& result) {
        auto tokenPrefix = TokenPrefix(token);

        TReadGuard rg(Mutex);
        if (FindSessionIn(*SessionsPresent, tokenPrefix, result)) {
            return NThreading::MakeFuture(true);
        }

        if (FindSessionIn(*SessionsPast, tokenPrefix, result)) {
            return NThreading::MakeFuture(true);
        }

        if (FindSessionIn(*SessionsFuture, tokenPrefix, result)) {
            return NThreading::MakeFuture(true);
        }

        return NThreading::MakeFuture(false);
    }

    NThreading::TFuture<bool> TCaptchaFallbackSessionStorage::StoreSessionInfo(TStringBuf, const TCaptchaSessionInfo&) {
        return NThreading::MakeFuture(true);
    }

    NThreading::TFuture<void> TCaptchaFallbackSessionStorage::DropSession(TStringBuf) {
        return NThreading::MakeFuture();
    }

    TCaptchaFallbackSessionStorage::~TCaptchaFallbackSessionStorage() {
        StopEvent.Signal();
        RotateThread.Join();
    }

    void TCaptchaFallbackSessionStorage::PrepareSessions(ui64 seed, TSessionsContainer& container) {
        // The idea is to have all fallback session lists synchronized over all backends
        // by setting up deterministic RNG with equal initial states.
        TDeterministicRng rng(seed);

        TCaptchaItemsStorageRouter tempRouter(false);
        tempRouter.RegisterStorage(ECaptchaItemsStorageId::Fallback, FallbackItemsStorage);
        TCaptchaSessionFactory sessFactory(Config, Stats, tempRouter);

        // Iterate over session counter range first and ctype/vtype combinations after that
        // to make sure increasing session count will not affect existing sessions
        // by altering order of RNG calls
        for (auto _ : xrange(Config.GetFallback().GetSessionsPerUniqueTypeCombo())) {
            Y_UNUSED(_);
            for (auto i : xrange(Config.CaptchaTypesSize())) {
                TString ctypeName = Config.GetCaptchaTypes(i).GetName();

                for (auto j : xrange(Config.VoiceCaptchaTypesSize())) {
                    TString vtypeName = Config.GetVoiceCaptchaTypes(j).GetName();

                    PrepareSessionForTypeCombo(&rng, ctypeName, vtypeName, sessFactory, container);
                }
            }
        }
    }

    void TCaptchaFallbackSessionStorage::PrepareSessionForTypeCombo(IRng* rng, const TString& ctypeName, const TString& vtypeName, TCaptchaSessionFactory& sessFactory, TSessionsContainer& container) {
        Y_ASSERT(rng);
        auto& tokenPrefixes = container.TokenPrefixesByTypeCombos[std::make_pair(ctypeName, vtypeName)];
        const auto& tokenPrefix = BuildToken(rng, FallbackTokenTag, TStringBuf(), TokenLength - RandomTokenPartLength);
        tokenPrefixes.push_back(tokenPrefix);

        auto& sessionInfo = container.PreparedSessions[tokenPrefix];

        TCaptchaSessionRequest request(ctypeName, vtypeName, -1, TInstant::Zero());
        sessFactory.MakeSession(request, ECaptchaItemsStorageId::Fallback, rng, sessionInfo);
        sessionInfo.Fallback = true;
    }

    void TCaptchaFallbackSessionStorage::RotateSessions() {
        THolder<TSessionsContainer> newSessions = MakeHolder<TSessionsContainer>();
        THolder<TSessionsContainer> oldSessions;
        ui64 seed = TimeCurrent + TimeInterval * 2 + Config.GetFallback().GetSeed();
        PrepareSessions(seed, *newSessions);

        DEBUG_LOG << "Rotating fallback sessions (TimeCurrent=" << TimeCurrent << "; seed=" << seed << ")" << Endl;

        {
            TWriteGuard wg(Mutex);
            oldSessions = std::move(SessionsPast); // oldSessions will be destroyed only on exit, outside the critical section
            SessionsPast = std::move(SessionsPresent);
            SessionsPresent = std::move(SessionsFuture);
            SessionsFuture = std::move(newSessions);
            TimeCurrent += TimeInterval;
        }
    }

    bool TCaptchaFallbackSessionStorage::FindSessionIn(TSessionsContainer& container, TStringBuf tokenPrefix, TCaptchaSessionInfo& result) {
        auto iter = container.PreparedSessions.find(tokenPrefix);
        if (iter != container.PreparedSessions.end()) {
            result = iter->second;
            return true;
        }
        return false;
    }

    void* TCaptchaFallbackSessionStorage::RotateLoop(void* ptr) {
        TCaptchaFallbackSessionStorage* thisptr = reinterpret_cast<TCaptchaFallbackSessionStorage*>(ptr);
        while (true) {
            ui64 timeCurrent;
            {
                TReadGuard rg(thisptr->Mutex);
                timeCurrent = thisptr->TimeCurrent;
            }
            TInstant nextRotate = TInstant::Seconds(timeCurrent + thisptr->TimeInterval);
            DEBUG_LOG << "Next fallback sessions rotation: " << nextRotate << Endl;
            if (thisptr->StopEvent.WaitD(nextRotate)) {
                return nullptr;
            }

            try {
                thisptr->RotateSessions();
            } catch (...) {
                FATAL_LOG << "Error in fallback session rotation loop (TimeCurent=" << timeCurrent << ")" << Endl;
                throw;
            }
        }
    }
}
