#pragma once

#include "captchatype.h"
#include "config.h"
#include "fallback_items_storage.h"
#include "random.h"
#include "session_factory.h"
#include "session_storage.h"
#include "stats.h"

#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/system/rwlock.h>
#include <util/system/thread.h>
#include <util/system/event.h>

namespace NCaptchaServer {
    class TCaptchaFallbackSessionStorage: public ICaptchaSessionStorage {
    public:
        TCaptchaFallbackSessionStorage(const TCaptchaConfig& config, TCaptchaStats& stats, TCaptchaFallbackItemsStorage* fallbackItems);

        char TokenTag() override {
            return FallbackTokenTag;
        }
        NThreading::TFuture<TString> CreateSession(const TCaptchaSessionRequest& request, TCaptchaSessionInfo& info) override;
        NThreading::TFuture<bool> LoadSessionInfo(TStringBuf token, TCaptchaSessionInfo& result) override;
        NThreading::TFuture<bool> StoreSessionInfo(TStringBuf token, const TCaptchaSessionInfo& info) override;
        NThreading::TFuture<void> DropSession(TStringBuf token) override;

        ~TCaptchaFallbackSessionStorage();

        template <typename Callback>
        void TraverseSessions(Callback callback, bool past, bool present, bool future) {
            if (past) {
                TraverseSessions(callback, *SessionsPast);
            }
            if (present) {
                TraverseSessions(callback, *SessionsPresent);
            }
            if (future) {
                TraverseSessions(callback, *SessionsFuture);
            }
        }

    private:
        struct TSessionsContainer {
            THashMap<std::pair<TString, TString>, TDeque<TString>> TokenPrefixesByTypeCombos;
            THashMap<TString, TCaptchaSessionInfo> PreparedSessions;
        };

    private:
        static char FallbackTokenTag;

    private:
        TStringBuf TokenPrefix(TStringBuf token) const {
            return token.Head(TokenLength - RandomTokenPartLength);
        }
        void PrepareSessions(ui64 seed, TSessionsContainer& container);
        void PrepareSessionForTypeCombo(IRng* rng, const TString& ctypeName, const TString& vtypeName, TCaptchaSessionFactory& sessFactory, TSessionsContainer& container);
        void RotateSessions();
        bool FindSessionIn(TSessionsContainer& container, TStringBuf tokenPrefix, TCaptchaSessionInfo& result);
        static void* RotateLoop(void* ptr);

        template <typename Callback>
        void TraverseSessions(Callback callback, TSessionsContainer& container) {
            TReadGuard rg(Mutex);
            for (auto it : container.PreparedSessions) {
                callback(it.first, it.second);
            }
        }

    private:
        TCaptchaConfig Config;
        TCaptchaStats& Stats;
        TCaptchaFallbackItemsStorage* FallbackItemsStorage;

        size_t RandomTokenPartLength;
        ui64 TimeInterval;

        THashMap<TString, TString> CaptchaTypeAliases;
        THashMap<TString, TString> VoiceCaptchaTypeAliases;

        TRWMutex Mutex;
        ui64 TimeCurrent;
        THolder<TSessionsContainer> SessionsPast;
        THolder<TSessionsContainer> SessionsPresent;
        THolder<TSessionsContainer> SessionsFuture;

        TManualEvent StopEvent;
        TThread RotateThread;
    };
}
