#include "handler_answer.h"

#include "items_storage.h"
#include "session_storage.h"

#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/json/writer/json.h>

#include <library/cpp/cgiparam/cgiparam.h>

namespace NCaptchaServer {
    namespace {
        struct THandlerAnswerState {
            bool Error = false;
            TStringBuf ErrorMessage;
            TString Answer;
            bool Fallback = false;

            TString Token;
            bool Voice = false;
            bool Json = false;

            TString RemoteHost;
            TString Ip;
        };
    }

    static THttpResponse MakeResponse(const THandlerAnswerState& state) {
        if (!state.Json) {
            TString result;
            TStringOutput so(result);
            NJsonWriter::TBuf json(NJsonWriter::HEM_DONT_ESCAPE_HTML, &so);
            json.BeginObject();
            if (state.Error) {
                json.WriteKey("error").WriteString(state.ErrorMessage);
            } else {
                if (state.Voice) {
                    json.WriteKey("voice").WriteInt(1);
                }
                json.WriteKey("answer").WriteString(state.Answer);
            }
            if (state.Fallback) {
                json.WriteKey("fallback").WriteString("1");
            }
            json.WriteKey("json").WriteString("1");
            json.EndObject();

            THttpResponse response(HTTP_OK);
            response.AddHeader("Content-Type", "application/json; charset=utf-8");
            response.SetContent(result);
            return response;
        } else {
            TString result;
            TStringOutput so(result);

            so << "<?xml version='1.0'?>\n";
            if (state.Error) {
                so << "<answer error=\"" << state.ErrorMessage << "\"></answer>";
            } else {
                if (state.Fallback) {
                    so << "<answer fallback='1'>" << state.Answer << "</answer>";
                } else {
                    so << "<answer>" << state.Answer << "</answer>";
                }
            }

            THttpResponse response(HTTP_OK);
            response.AddHeader("Content-Type", "text/xml; charset=utf-8");
            response.SetContent(result);
            return response;
        }
    }

    static void GetAnswer(THandlerAnswerState& state, const TCaptchaSessionInfo& sessionInfo, ICaptchaType* captchaType) {
        if (state.Voice) {
            const auto& sessionMetadata = sessionInfo.Metadata.GetMapSafe();
            if (sessionMetadata.contains("voice_metadata")) {
                state.Answer = sessionMetadata.at("voice_metadata").GetMapSafe().at("answer").GetStringSafe();
            } else {
                state.Error = true;
                state.ErrorMessage = TStringBuf("voice wasnt requested");
            }
        } else {
            if (!captchaType->GetAnswer(sessionInfo.Type, sessionInfo, state.Answer)) {
                state.Error = true;
                state.ErrorMessage = TStringBuf("not implemented");
            }
        }
    }

    NThreading::TFuture<THttpResponse> THandlerAnswer::HandleRequest(TRequestInfo& reqInfo) {
        TParsedHttpFull req(reqInfo.HttpInput.FirstLine());
        TCgiParameters params(req.Cgi);

        TAtomicSharedPtr<THandlerAnswerState> statePtr(new THandlerAnswerState);
        TAtomicSharedPtr<TCaptchaSessionInfo> sessionInfoPtr(new TCaptchaSessionInfo);

        reqInfo.Token = params.Get("key");

        statePtr->Token = reqInfo.Token;
        statePtr->Json = EqualToOneOf(params.Get("json"), "", "0");
        statePtr->Voice = !EqualToOneOf(params.Get("voice"), "", "0");
        statePtr->RemoteHost = reqInfo.RemoteHost;
        statePtr->Ip = params.Get("ip");

        ICaptchaSessionStorage* sessionStorage = nullptr;

        if (statePtr->Token) {
            sessionStorage = SessionStorageRouter.GetStorageByToken(statePtr->Token);
        }

        if (!sessionStorage) {
            statePtr->Error = true;
            statePtr->ErrorMessage = TStringBuf("not found");
            return NThreading::MakeFuture(MakeResponse(*statePtr));
        }

        if (SessionStorageRouter.IsFallbackStorage(sessionStorage) && Config.GetFallback().GetStatusInAnswerResponse()) {
            statePtr->Fallback = true;
        }

        auto cont1 = [statePtr, sessionInfoPtr, sessionStorage, this](const NThreading::TFuture<bool>& fsessionLoadSuccess) {
            if (!fsessionLoadSuccess.GetValue()) {
                statePtr->Error = true;
                statePtr->ErrorMessage = TStringBuf("not found");
                return NThreading::MakeFuture(MakeResponse(*statePtr));
            }

            GetAnswer(*statePtr, *sessionInfoPtr, Server->SessionFactory.GetCaptchaType(sessionInfoPtr->Type));
            if (statePtr->Error) {
                return NThreading::MakeFuture(MakeResponse(*statePtr));
            }

            sessionInfoPtr->Metadata["requested_data"] = 1;
            sessionInfoPtr->Metadata["answer_server_ip"] = statePtr->RemoteHost;
            if (statePtr->Ip) {
                sessionInfoPtr->Metadata["answer_client_ip"] = statePtr->Ip;
            }
            if (!sessionInfoPtr->Metadata.GetMapSafe().contains("answer_timestamp")) {
                sessionInfoPtr->Metadata["answer_timestamp"] = Now().MilliSeconds();
            }

            auto cont2 = [statePtr](const NThreading::TFuture<bool>& storeResult) {
                storeResult.GetValue();
                return MakeResponse(*statePtr);
            };
            return sessionStorage->StoreSessionInfo(statePtr->Token, *sessionInfoPtr).Apply(cont2);
        };
        return sessionStorage->LoadSessionInfo(statePtr->Token, *sessionInfoPtr).Apply(cont1);
    }
}
