#include "handler_check.h"

#include "session_storage.h"

#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/json/writer/json.h>

#include <library/cpp/cgiparam/cgiparam.h>

#include <time.h>

namespace NCaptchaServer {
    static bool IsValidAnswer(const TString& answer) {
        for (auto c : answer) {
            if (c == '\n') {
                return false;
            } else if (c == '\t') {
                return false;
            }
        }
        return true;
    }

    static THttpResponse MakeResponse(const THandlerCheckState& state) {
        if (state.Json) {
            TString result;
            TStringOutput so(result);
            NJsonWriter::TBuf json(NJsonWriter::HEM_DONT_ESCAPE_HTML, &so);
            json.BeginObject();
            json.WriteKey("status").WriteString(state.Status);
            if (state.Error) {
                if (state.ErrorDesc) {
                    json.WriteKey("error_desc").WriteString(state.ErrorDesc);
                }
                json.WriteKey("error").WriteString(state.ErrorMessage);
            }
            if (state.Oldstyle) {
                json.WriteKey("oldstyle").WriteInt(1);
            }
            if (state.Client) {
                if (state.Answer) {
                    json.WriteKey("answer").WriteString(state.Answer);
                }
                if (state.VoiceAnswer) {
                    json.WriteKey("voice_answer").WriteString(state.VoiceAnswer);
                }
                // TODO: от Answer и VoiceAnswer будем избавляться
                if (state.SessionMetadata.IsMap()) {
                    json.WriteKey("session_metadata").WriteJsonValue(&state.SessionMetadata);
                }
            }
            json.WriteKey("json").WriteString("1");
            json.EndObject();

            THttpResponse response(HTTP_OK);
            response.AddHeader("Content-Type", "application/json; charset=utf-8");
            response.SetContent(result);
            return response;
        } else {
            TString result;
            TStringOutput so(result);

            if (state.Oldstyle) {
                so << "<image_check>" << state.Status << "</image_check>";
            } else {
                so << "<?xml version='1.0'?>\n";
                so << "<image_check";
                if (state.Error) {
                    so << " error=\"" << state.ErrorMessage << "\"";
                    if (state.ErrorDesc) {
                        so << " error_desc=\"" << state.ErrorDesc << "\"";
                    }
                }
                so << ">" << state.Status << "</image_check>";
            }

            THttpResponse response(HTTP_OK);
            response.AddHeader("Content-Type", "text/xml; charset=utf-8");
            response.SetContent(result);
            return response;
        }
    }

    NThreading::TFuture<THttpResponse> THandlerCheck::HandleRequest(TRequestInfo& reqInfo) {
        TParsedHttpFull req(reqInfo.HttpInput.FirstLine());
        TCgiParameters params(req.Cgi);

        TAtomicSharedPtr<THandlerCheckState> statePtr(new THandlerCheckState);
        TAtomicSharedPtr<TCaptchaSessionInfo> sessionInfoPtr(new TCaptchaSessionInfo);

        reqInfo.Token = params.Get("key");

        statePtr->Token = reqInfo.Token;
        statePtr->Client = params.Get("client");
        statePtr->Rep = params.Get("rep");
        if (!statePtr->Rep) {
            statePtr->Rep = params.Get("res");
        }
        statePtr->Json = !EqualToOneOf(params.Get("json"), "", "0");
        statePtr->Ip = params.Get("ip");
        statePtr->RemoteHost = reqInfo.RemoteHost;
        statePtr->Oldstyle = params.Get("style") == "old";
        statePtr->Status = TStringBuf("failed");
        statePtr->LogStatus = TStringBuf("fail");

        if (!statePtr->Token) {
            statePtr->Error = true;
            statePtr->ErrorMessage = TStringBuf("not found");
        } else if (!statePtr->Rep) {
            statePtr->Error = true;
            statePtr->ErrorMessage = TStringBuf("no user res(ponse)");
        }

        ICaptchaSessionStorage* sessionStorage = SessionStorageRouter.GetStorageByToken(statePtr->Token);
        if (!sessionStorage) {
            statePtr->Error = true;
            statePtr->ErrorMessage = TStringBuf("not found");
        }

        if (statePtr->Error) {
            return NThreading::MakeFuture(MakeResponse(*statePtr));
        }

        TDuration timeout = TDuration::Seconds(Config.GetSessionTimeoutSeconds());
        auto cont1 = [statePtr, sessionInfoPtr, sessionStorage, this](const NThreading::TFuture<bool>& fsessionLoadSuccess) {
            if (!fsessionLoadSuccess.GetValue()) {
                statePtr->Error = true;
                statePtr->ErrorMessage = TStringBuf("not found");
                return NThreading::MakeFuture(MakeResponse(*statePtr));
            }

            if (SessionStorageRouter.IsFallbackStorage(sessionStorage)) {
                Server->Stats.PushSignal(ESignals::StorageFallbackRequestsCheck);
            }

            if (!sessionInfoPtr->Metadata.GetMapSafe().contains("requested_data") && !sessionInfoPtr->Fallback) {
                statePtr->Error = true;
                statePtr->ErrorMessage = TStringBuf("not found");
                statePtr->ErrorDesc = TStringBuf("image not allocated, go get image on /image first");
                return NThreading::MakeFuture(MakeResponse(*statePtr));
            }

            sessionInfoPtr->Metadata["check_server_ip"] = statePtr->RemoteHost;
            if (statePtr->Ip) {
                sessionInfoPtr->Metadata["check_client_ip"] = statePtr->Ip;
            }
            if (!sessionInfoPtr->Metadata.GetMapSafe().contains("check_timestamp")) {
                sessionInfoPtr->Metadata["check_timestamp"] = Now().MilliSeconds();
            }

            NThreading::TFuture<void> updateSession;
            if (CheckAnswer(*sessionInfoPtr, statePtr)) {
                statePtr->LogStatus = statePtr->Status = TStringBuf("ok");
                sessionInfoPtr->Checks -= 1;
                if (sessionInfoPtr->Checks <= 0) {
                    updateSession = sessionStorage->DropSession(statePtr->Token);
                } else {
                    updateSession = sessionStorage->StoreSessionInfo(statePtr->Token, *sessionInfoPtr).IgnoreResult();
                }
            } else {
                updateSession = sessionStorage->DropSession(statePtr->Token);
            }

            auto cont2 = [statePtr, sessionInfoPtr, this](const NThreading::TFuture<void>& updateResult) {
                updateResult.GetValue();

                ICaptchaType* ctype = Server->GetCaptchaType(sessionInfoPtr->Type);
                ctype->PrepareLoggedUserAnswer(sessionInfoPtr->Type, *sessionInfoPtr, statePtr->Rep);
                if (IsValidAnswer(statePtr->Rep)) {
                    MakeLogEntry(statePtr->LogStatus, statePtr->Token, *sessionInfoPtr, statePtr->Rep);
                    PushStats(statePtr->Status == "ok"sv, *sessionInfoPtr);
                } else {
                    statePtr->Error = true;
                    statePtr->ErrorMessage = TStringBuf("invalid characters in the user answer");
                }
                return MakeResponse(*statePtr);
            };
            return updateSession.Apply(cont2);
        };

        return sessionStorage->LoadAndValidateSessionInfo(statePtr->Token, *sessionInfoPtr, timeout).Apply(cont1);
    }

    bool THandlerCheck::CheckAnswer(const TCaptchaSessionInfo& sessInfo, TAtomicSharedPtr<THandlerCheckState> statePtr) {
        TStringBuf userAnswer = statePtr->Rep;
        ICaptchaType* ctype = Server->GetCaptchaType(sessInfo.Type);
        bool result = false;
        ctype->GetAnswer(sessInfo.Type, sessInfo, statePtr->Answer);
        if (ctype->CheckAnswer(sessInfo.Type, sessInfo, userAnswer)) {
            result = true;
        }

        const auto& sessionMetadata = sessInfo.Metadata.GetMapSafe();
        if (sessionMetadata.contains("voice_metadata")) {
            const auto& voiceAnswer = sessionMetadata.at("voice_metadata").GetMapSafe().at("answer").GetStringSafe();
            statePtr->VoiceAnswer = voiceAnswer;
            if (userAnswer == voiceAnswer) {
                result = true;
            }
        }

        statePtr->SessionMetadata = sessInfo.Metadata;

        return result;
    }

    static TMaybe<TDuration> GetAnswerTime(const TCaptchaSessionInfo& sessInfo) {
        if (sessInfo.Fallback) {
            return Nothing();
        }

        const auto& sessionMetadata = sessInfo.Metadata.GetMapSafe();

        auto image_ts = sessionMetadata.find("image_timestamp");
        auto check_ts = sessionMetadata.find("check_timestamp");

        if (image_ts != sessionMetadata.end() && check_ts != sessionMetadata.end()) {
            TInstant from = TInstant::MilliSeconds(image_ts->second.GetUIntegerSafe());
            TInstant to = TInstant::MilliSeconds(check_ts->second.GetUIntegerSafe());
            return TMaybe<TDuration>(to - from);
        }

        return Nothing();
    }

    void THandlerCheck::MakeLogEntry(TStringBuf status, const TString& token, const TCaptchaSessionInfo& sessInfo, TStringBuf userAnswer) {
        if (sessInfo.Fallback && !Config.GetFallback().GetEnableChecksLog()) {
            return;
        }

        ICaptchaType* ctype = Server->GetCaptchaType(sessInfo.Type);

        TString line;
        TStringOutput so(line);

        so << "tskv\ttskv_format=captcha-checks-log"
           << "\ttimestamp=" << GetTimestamp()
           << "\ttimezone=+0000"
           << "\tcheck=" << status
           << "\tKEY=" << token
           << "\tchecks=" << sessInfo.Checks;

        if (sessInfo.Fallback) {
            so << "\tfallback=true";
        }

        TString answer;
        if (ctype->GetAnswer(sessInfo.Type, sessInfo, answer)) {
            so << "\tanswer=" << answer;
        }

        const auto& sessionMetadata = sessInfo.Metadata.GetMapSafe();

        so << "\trep=" << userAnswer
           << "\ttype=" << sessInfo.Type
           << "\tvtype=" << sessionMetadata.at("voice_type").GetStringSafe()
           << "\tvanswer=" << sessionMetadata.at("voice_metadata").GetMapSafe().at("answer").GetStringSafe();

        auto addFromMetadata = [&so, &sessionMetadata](const char* mdkey, const char* logkey) {
            if (sessionMetadata.contains(mdkey)) {
                so << "\t" << logkey << "=" << sessionMetadata.at(mdkey).GetStringSafe();
            }
        };

        addFromMetadata("image_client_ip", "ip_i");
        addFromMetadata("check_client_ip", "ip_c");
        addFromMetadata("check_server_ip", "ip_c*");
        addFromMetadata("answer_client_ip", "ip_a");
        addFromMetadata("answer_server_ip", "ip_a*");

        auto answerTime = GetAnswerTime(sessInfo);
        if (answerTime) {
            so << "\ttimer_c=" << answerTime->MilliSeconds();
        }

        so << "\tversion=3" << Endl;

        ChecksLog << line;
    }

    void THandlerCheck::PushStats(bool status, const TCaptchaSessionInfo& sessInfo) {
        ESignals checksSig, answerTimeSig;
        ETypeSegmentedSignals checksTypedSig, answerTimeTypedSig;

        if (status) {
            checksSig = ESignals::ChecksOk;
            answerTimeSig = ESignals::AnswerTimeMsOk;
            checksTypedSig = ETypeSegmentedSignals::ChecksOk;
            answerTimeTypedSig = ETypeSegmentedSignals::AnswerTimeMsOk;
        } else {
            checksSig = ESignals::ChecksFail;
            answerTimeSig = ESignals::AnswerTimeMsFail;
            checksTypedSig = ETypeSegmentedSignals::ChecksFail;
            answerTimeTypedSig = ETypeSegmentedSignals::AnswerTimeMsFail;
        }

        Server->Stats.PushSignal(checksSig);
        Server->Stats.PushSignal(checksTypedSig, sessInfo.Type);

        auto answerTime = GetAnswerTime(sessInfo);
        if (answerTime) {
            double answerTimeMs = answerTime->MilliSeconds();
            Server->Stats.PushSignal(answerTimeSig, answerTimeMs);
            Server->Stats.PushSignal(answerTimeTypedSig, sessInfo.Type, answerTimeMs);
        }
    }
}
