#include "handler_image.h"

#include "items_storage.h"
#include "session_storage.h"

#include <library/cpp/http/misc/parsed_request.h>

#include <library/cpp/cgiparam/cgiparam.h>

namespace NCaptchaServer {
    NThreading::TFuture<THttpResponse> THandlerImage::HandleRequest(TRequestInfo& reqInfo) {
        TParsedHttpFull req(reqInfo.HttpInput.FirstLine());
        TCgiParameters params(req.Cgi);

        TAtomicSharedPtr<TCaptchaSessionInfo> sessionInfoPtr(new TCaptchaSessionInfo);
        const TString& token = params.Get("key");
        reqInfo.Token = token;

        if (!token) {
            return NThreading::MakeFuture(THttpResponse(HTTP_NOT_FOUND));
        }

        ICaptchaSessionStorage* sessionStorage = SessionStorageRouter.GetStorageByToken(token);
        if (!sessionStorage) {
            return NThreading::MakeFuture(THttpResponse(HTTP_NOT_FOUND));
        }

        auto cont1 = [reqInfo, sessionInfoPtr, token, sessionStorage, this](const NThreading::TFuture<bool>& result) {
            if (!result.GetValue()) {
                return NThreading::MakeFuture(THttpResponse(HTTP_NOT_FOUND));
            }

            if (SessionStorageRouter.IsFallbackStorage(sessionStorage)) {
                Server->Stats.PushSignal(ESignals::StorageFallbackRequestsImage);
            }

            sessionInfoPtr->Metadata["requested_data"] = 1;
            sessionInfoPtr->Metadata["image_client_ip"] = reqInfo.RemoteHost;
            if (!sessionInfoPtr->Metadata.GetMapSafe().contains("image_timestamp")) {
                sessionInfoPtr->Metadata["image_timestamp"] = Now().MilliSeconds();
            }

            TString type = sessionInfoPtr->Type;
            auto storeFuture = sessionStorage->StoreSessionInfo(token, *sessionInfoPtr);
            auto imageFuture = Server->GetCaptchaType(type)->HandleGetImage(type, *sessionInfoPtr, reqInfo);

            auto cont2 = [imageFuture](const NThreading::TFuture<void>& waitResult) {
                waitResult.GetValue();
                return imageFuture.GetValue();
            };
            return NThreading::WaitExceptionOrAll(storeFuture.IgnoreResult(), imageFuture.IgnoreResult()).Apply(cont2);
        };

        return sessionStorage->LoadSessionInfo(token, *sessionInfoPtr).Apply(cont1);
    }
}
