#include "handler_voice.h"

#include "items_storage.h"
#include "session_storage.h"

#include <library/cpp/http/misc/parsed_request.h>

#include <library/cpp/cgiparam/cgiparam.h>

namespace NCaptchaServer {
    NThreading::TFuture<THttpResponse> THandlerVoice::HandleRequest(TRequestInfo& reqInfo) {
        TParsedHttpFull req(reqInfo.HttpInput.FirstLine());
        TCgiParameters params(req.Cgi);

        TAtomicSharedPtr<TCaptchaSessionInfo> sessionInfoPtr(new TCaptchaSessionInfo);
        const TString& token = params.Get("key");
        reqInfo.Token = token;

        if (!token) {
            return NThreading::MakeFuture(THttpResponse(HTTP_NOT_FOUND));
        }

        ICaptchaSessionStorage* sessionStorage = SessionStorageRouter.GetStorageByToken(token);
        if (!sessionStorage) {
            return NThreading::MakeFuture(THttpResponse(HTTP_NOT_FOUND));
        }

        auto cont1 = [reqInfo, sessionInfoPtr, token, sessionStorage, this](const NThreading::TFuture<bool>& result) {
            if (!result.GetValue()) {
                return NThreading::MakeFuture(THttpResponse(HTTP_NOT_FOUND));
            }

            if (SessionStorageRouter.IsFallbackStorage(sessionStorage)) {
                Server->Stats.PushSignal(ESignals::StorageFallbackRequestsVoice);
            }

            const auto& sessionMetadata = sessionInfoPtr->Metadata.GetMapSafe();
            if (!sessionMetadata.contains("voice_key")) {
                return NThreading::MakeFuture(THttpResponse(HTTP_NOT_FOUND));
            }

            TAtomicSharedPtr<TString> contentPtr(new TString());
            TCaptchaItemKey key(sessionMetadata.at("voice_key"));
            TString contentType = sessionMetadata.at("voice_metadata").GetMapSafe().at("content_type").GetStringSafe();

            sessionInfoPtr->Metadata["requested_data"] = 1;

            ICaptchaItemsStorage* itemsStorage = ItemsStorageRouter.GetStorage(sessionInfoPtr->ItemsStorage);
            auto storeFuture = sessionStorage->StoreSessionInfo(token, *sessionInfoPtr);
            auto voiceFuture = itemsStorage->LoadItemData(key, *contentPtr);

            auto cont2 = [voiceFuture, contentPtr, contentType](const NThreading::TFuture<void>& waitResult) {
                waitResult.GetValue();

                if (!voiceFuture.GetValue()) {
                    return THttpResponse(HTTP_INTERNAL_SERVER_ERROR);
                }

                THttpResponse response(HTTP_OK);
                response.SetContent(*contentPtr);
                response.AddHeader("Content-Type", contentType);
                return response;
            };
            return NThreading::WaitExceptionOrAll(storeFuture.IgnoreResult(), voiceFuture.IgnoreResult()).Apply(cont2);
        };

        return sessionStorage->LoadSessionInfo(token, *sessionInfoPtr).Apply(cont1);
    }
}
