#pragma once

#include "captchatype.h"
#include "config.h"
#include "random.h"
#include "session_storage.h"
#include "stats.h"

#include <util/datetime/base.h>
#include <util/generic/xrange.h>
#include <util/system/mutex.h>

namespace NCaptchaServer {
    namespace NPrivate {
        class TScopedTimer {
        public:
            TScopedTimer(TCaptchaStats& stats, ESignals signal)
                : Stats(stats)
                , Signal(signal)
            {
                Start = Now();
            }

            ~TScopedTimer() {
                TDuration duration = Now() - Start;
                Stats.PushSignal(Signal, duration.MilliSeconds());
            }

        private:
            TCaptchaStats& Stats;
            ESignals Signal;
            TInstant Start;
        };
    }

    class TResourceAvailabilityMonitor {
    private:
        struct TState {
            TMutex Mutex;
            bool Alive = false;
        };

        friend class TResourceAccessorGuard;

    public:
        TResourceAvailabilityMonitor()
            : State(new TState)
        {
        }

        void Create() {
            with_lock (State->Mutex) {
                Y_ASSERT(!State->Alive);
                State->Alive = true;
            }
        }

        void Destroy() {
            with_lock (State->Mutex) {
                Y_ASSERT(State->Alive);
                State->Alive = false;
            }
        }

    private:
        TAtomicSharedPtr<TState> State;
    };

    class TResourceProviderGuard: public TNonCopyable {
    public:
        TResourceProviderGuard(const TResourceAvailabilityMonitor& monitor)
            : Monitor(monitor)
        {
            Monitor.Create();
        }

        ~TResourceProviderGuard() {
            Monitor.Destroy();
        }

    private:
        TResourceAvailabilityMonitor Monitor;
    };

    class TResourceAccessorGuard: public TNonCopyable {
    public:
        TResourceAccessorGuard(const TResourceAvailabilityMonitor& monitor)
            : Monitor(monitor)
            , Guard(monitor.State->Mutex)
        {
        }

        operator bool() const {
            return Monitor.State->Alive;
        }

    private:
        TResourceAvailabilityMonitor Monitor;
        TGuard<TMutex> Guard;
    };

    template <typename Callable>
    auto TimedRun(Callable func, TCaptchaStats& stats, ESignals signal) -> decltype(func()) {
        NPrivate::TScopedTimer st(stats, signal);
        return func();
    }

    template <typename TTypeList>
    void FillTypeAliases(const TTypeList& typelist, THashMap<TString, TString>& result) {
        for (const auto& typedesc : typelist) {
            const auto& name = typedesc.GetName();
            Y_ENSURE(result.insert({name, name}).second);
            for (const auto& alias : typedesc.GetAliases()) {
                Y_ENSURE(result.insert({alias, name}).second);
            }
        }
    }

    inline const TString& GetWithDefault(const THashMap<TString, TString>& map, TStringBuf key) {
        auto iter = map.find(key);
        if (iter != map.end()) {
            return iter->second;
        } else {
            return map.at("");
        }
    }

    THolder<ICaptchaType> MakeCaptchaType(const TCaptchaConfig& config, TCaptchaStats& stats, TCaptchaItemsStorageRouter& itemsStorageRouter, const TCaptchaConfig::TCaptchaTypeOptions& options);
    void RandomAlNumString(IRng* rng, size_t length, TStringOutput& output);
    TString BuildToken(IRng* rng, char storageTag, TStringBuf prefix = "", size_t length = TokenLength);

    static inline TString GetTimestamp() {
        char timebuf[32];
        time_t now;
        struct tm curtime;
        time(&now);
        gmtime_r(&now, &curtime);
        strftime(timebuf, sizeof(timebuf), "%F %T", &curtime);
        return timebuf;
    }
}
