#include "http_server.h"

#include "fallback_session_storage.h"

#include "handler_generate.h"
#include "handler_image.h"
#include "handler_voice.h"
#include "handler_check.h"
#include "handler_answer.h"
#include "handler_static.h"
#include "handler_randomitem.h"

#include "helpers.h"
#include "captchatype_ocr.h"
#include "captchatype_localization.h"

#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/json/writer/json.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/logger/thread.h>
#include <library/cpp/logger/file.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <security/ant-secret/snooper/cpp/logger/masked.h>
#include <util/string/type.h>
#include <util/generic/xrange.h>

namespace NCaptchaServer {
    class TCaptchaRequestReplier: public TRequestReplier {
    public:
        TCaptchaRequestReplier(TCaptchaHttpServer* server)
            : Server(server)
        {
        }

        THttpResponse ExceptionResponse(std::exception_ptr exc);
        void FinishProcessing(const TRequestReplier::TReplyParams& params, const THttpResponse& resp);
        void MakeAccessLogEntry(const THttpResponse& resp);
        bool DoReply(const TRequestReplier::TReplyParams& params) override;

    private:
        TCaptchaHttpServer* Server;
        THolder<TRequestInfo> RequestInfo;
        NThreading::TFuture<THttpResponse> Response;
        bool Delayed = false;
        TInstant StartTime;
        TString HandlerName;
    };

    THttpResponse TCaptchaRequestReplier::ExceptionResponse(std::exception_ptr exc) {
        TParsedHttpFull req(RequestInfo->HttpInput.FirstLine());
        TString errorMessage;

        try {
            std::rethrow_exception(exc);
        } catch (const std::exception& ex) {
            errorMessage = ex.what();
        } catch (...) {
            errorMessage = "Unknown exception";
        }

        Server->Stats.PushSignal(ESignals::TotalExceptions);
        ERROR_LOG << "Unexpected error: " << errorMessage.Quote() << " while handling request " << TString{req.Request}.Quote() << Endl;

        {
            TReadGuard rg(Server->ConfigMutex);
            if (!Server->Config.GetDisplayInternalErrorDetails()) {
                errorMessage = "Internal error";
            }
        }

        return THttpResponse(HTTP_INTERNAL_SERVER_ERROR).SetContent(errorMessage);
    }

    void TCaptchaRequestReplier::FinishProcessing(const TRequestReplier::TReplyParams& params, const THttpResponse& resp) {
        MakeAccessLogEntry(resp);

        double responseTimeMs = (Now() - StartTime).MilliSeconds();
        Server->Stats.PushSignal(ESignals::ResponseTimingsMs, responseTimeMs);
        Server->Stats.PushSignal(EHandlerSegmentedSignals::ResponseTimingsMs, HandlerName, responseTimeMs);

        TimedRun([&]() { params.Output << resp; }, Server->Stats, ESignals::WriteResponseTimingsMs);
    }

    static size_t CalculateResponseSize(const THttpResponse& resp) {
        class TSizeCalculator: public IOutputStream {
        public:
            void DoWrite(const void*, size_t len) override {
                Size += len;
            }

            size_t GetSize() const {
                return Size;
            }

        private:
            size_t Size = 0;
        };

        TSizeCalculator sizecalc;
        resp.OutTo(sizecalc);
        return sizecalc.GetSize();
    }

    void TCaptchaRequestReplier::MakeAccessLogEntry(const THttpResponse& resp) {
        if (!Server->AccessLog) {
            return;
        }

        double processingTime = double((Now() - StartTime).MicroSeconds()) / 1000.0;

        time_t unixtime;
        time(&unixtime);

        TString line;
        TStringOutput so(line);
        NJsonWriter::TBuf json(NJsonWriter::HEM_UNSAFE, &so);

        TParsedHttpFull req(RequestInfo->HttpInput.FirstLine());

        json.BeginObject();
        json.WriteKey("unixtime").WriteULongLong(unixtime);
        json.WriteKey("request_time").WriteString(StartTime.ToString());
        json.WriteKey("ip").WriteString(RequestInfo->RealIp);
        json.WriteKey("source_addr").WriteString(RequestInfo->RemoteHost);
        json.WriteKey("url").WriteString(req.Request);
        json.WriteKey("status_code").WriteInt(resp.HttpCode());
        json.WriteKey("token").WriteString(RequestInfo->Token);
        json.WriteKey("processing_time_ms").WriteDouble(processingTime);
        json.WriteKey("response_size").WriteULongLong(CalculateResponseSize(resp));
        json.WriteKey("headers").BeginList();
        for (const auto& header : RequestInfo->HttpInput.Headers()) {
            json.BeginList();
            json.WriteString(header.Name());
            json.WriteString(header.Value());
            json.EndList();
        }
        json.EndList();
        json.EndObject();

        so << Endl;

        (*Server->AccessLog) << line;
    }

    bool TCaptchaRequestReplier::DoReply(const TRequestReplier::TReplyParams& params) {
        if (!Delayed) {
            StartTime = Now();
            TParsedHttpFull req(params.Input.FirstLine());

            RequestInfo = MakeHolder<TRequestInfo>(params.Input, Socket(), IsLocal());
            HandlerName = Server->HandlerNameFromPath(req.Path);
            if (!HandlerName) {
                INFO_LOG << "Not found: " << TString{req.Path}.Quote() << Endl;
                THttpResponse resp(HTTP_NOT_FOUND);
                MakeAccessLogEntry(resp);
                TimedRun([&]() { params.Output << resp; }, Server->Stats, ESignals::WriteResponseTimingsMs);
                return true;
            }

            IHandler* handler = Server->Handlers.at(HandlerName).Get();
            Server->Stats.PushSignal(ESignals::TotalRequests);
            Server->Stats.PushSignal(EHandlerSegmentedSignals::Requests, HandlerName);
            Server->RequestRateMonitor.RegisterRequest();

            try {
                Response = handler->HandleRequest(*RequestInfo);
            } catch (...) {
                THttpResponse resp = ExceptionResponse(std::current_exception());
                FinishProcessing(params, resp);
                return true;
            }
        }

        if (!Response.HasValue() && !Response.HasException()) {
            Delayed = true;
            auto callback = [this](const NThreading::TFuture<THttpResponse>&) {
                static_cast<IObjectInQueue*>(this)->Process(nullptr);
            };
            Response.Subscribe(callback);
            return false;
        }

        THttpResponse resp;
        try {
            resp = Response.GetValue();
        } catch (...) {
            resp = ExceptionResponse(std::current_exception());
        }
        FinishProcessing(params, resp);
        return true;
    }

    TClientRequest* TCaptchaHttpServer::CreateClient() {
        return new TCaptchaRequestReplier(this);
    }

    class THandlerPing: public IHandler {
    public:
        THandlerPing(const TCaptchaConfig& config)
            : Config(config)
        {
        }

        NThreading::TFuture<THttpResponse> HandleRequest(TRequestInfo&) override {
            THttpResponse response;

            if (Config.GetHttpServer().GetRejectPings()) {
                response.SetHttpCode(HTTP_INTERNAL_SERVER_ERROR);
            } else {
                response.SetHttpCode(HTTP_OK);
                response.SetContent("Pong\n");
            }
            return NThreading::MakeFuture(response);
        }

    private:
        TCaptchaConfig Config;
    };

    class THandlerItemsIndex: public IHandler {
    public:
        THandlerItemsIndex(TCaptchaItemsStorageRouter& itemsStorageRouter)
            : ItemsStorageRouter(itemsStorageRouter)
        {
        }

        NThreading::TFuture<THttpResponse> HandleRequest(TRequestInfo& reqInfo) override {
            if (!reqInfo.Local) {
                return NThreading::MakeFuture(THttpResponse(HTTP_FORBIDDEN));
            }

            TParsedHttpFull req(reqInfo.HttpInput.FirstLine());
            TCgiParameters params(req.Cgi);

            ICaptchaItemsStorage* itemsStorage = ItemsStorageRouter.GetStorage(params.Get("storage"));

            TString result;
            TStringOutput so(result);
            TCaptchaItemsIndex index;

            if (params.Get("type")) {
                itemsStorage->LoadIndex(params.Get("type"), index);
            } else {
                itemsStorage->LoadIndex(index);
            }

            so << "type\tversion\tid\tmetadata" << Endl;
            for (const auto& item : index) {
                so << item.Key.Type.Quote() << "\t"
                   << item.Key.Version.Quote() << "\t"
                   << item.Key.Id.Quote() << "\t"
                   << item.Metadata.GetStringRobust() << Endl;
            }

            THttpResponse response;
            response.SetHttpCode(HTTP_OK);
            response.SetHttpCode(HTTP_OK);
            response.SetContent(result);
            return NThreading::MakeFuture(response);
        }

    private:
        TCaptchaItemsStorageRouter& ItemsStorageRouter;
    };

    class THandlerPreloadItems: public IHandler {
    public:
        THandlerPreloadItems(TCaptchaItemsStorageRouter& itemsStorageRouter)
            : ItemsStorageRouter(itemsStorageRouter)
        {
        }

        NThreading::TFuture<THttpResponse> HandleRequest(TRequestInfo& reqInfo) override {
            if (!reqInfo.Local) {
                return NThreading::MakeFuture(THttpResponse(HTTP_FORBIDDEN));
            }

            TString result;
            TStringOutput so(result);
            TCaptchaItemsIndex index;
            size_t counter = 0;
            size_t totalSize = 0;

            ICaptchaItemsStorage* itemsStorage = ItemsStorageRouter.GetStorage("");
            itemsStorage->LoadIndex(index);
            TString value;

            for (const auto& item : index) {
                itemsStorage->LoadItemData(item.Key, value);
                ++counter;
                totalSize += value.size();
            }

            so << "Preloaded " << counter << " items (total size: " << totalSize << " bytes)" << Endl;
            THttpResponse response;
            response.SetHttpCode(HTTP_OK);
            response.SetContent(result);
            return NThreading::MakeFuture(response);
        }

    private:
        TCaptchaItemsStorageRouter& ItemsStorageRouter;
    };

    class THandlerFallbackList: public IHandler {
    public:
        THandlerFallbackList(TCaptchaSessionStorageRouter& sessionStorageRouter)
            : SessionFallback(sessionStorageRouter.GetFallbackStorage())
        {
        }

        NThreading::TFuture<THttpResponse> HandleRequest(TRequestInfo& reqInfo) override {
            if (!reqInfo.Local) {
                return NThreading::MakeFuture(THttpResponse(HTTP_FORBIDDEN));
            }

            if (!SessionFallback) {
                THttpResponse response;
                response.SetHttpCode(HTTP_BAD_REQUEST);
                response.SetContent("Fallback is disabled");
                return NThreading::MakeFuture(response);
            }

            TParsedHttpFull req(reqInfo.HttpInput.FirstLine());
            TCgiParameters params(req.Cgi);

            bool past = IsTrue(params.Get("past"));
            bool present = !params.Get("present") || IsTrue(params.Get("present"));
            bool future = IsTrue(params.Get("future"));

            TString result;
            TStringOutput so(result);

            auto callback = [&so](const TString& tokenPrefix, const TCaptchaSessionInfo& sessInfo) {
                so << tokenPrefix << "\t" << sessInfo.DebugStr() << Endl;
            };
            SessionFallback->TraverseSessions(callback, past, present, future);

            THttpResponse response;
            response.SetHttpCode(HTTP_OK);
            response.SetContent(result);
            return NThreading::MakeFuture(response);
        }

    private:
        TCaptchaFallbackSessionStorage* SessionFallback;
    };

    class THandlerSetFallbackState: public IHandler {
    public:
        THandlerSetFallbackState(TCaptchaFallbackStateAggregator& fallbackStateAggregator, bool activate)
            : FallbackStateAggregator(fallbackStateAggregator)
            , Activate(activate)
        {
        }

        NThreading::TFuture<THttpResponse> HandleRequest(TRequestInfo& reqinfo) override {
            if (!reqinfo.Local) {
                return NThreading::MakeFuture(THttpResponse(HTTP_FORBIDDEN));
            }

            FallbackStateAggregator.SetFallbackState(ECaptchaFallbackActivator::ConfigOrFallbackHandler, Activate);
            return NThreading::MakeFuture(THttpResponse(HTTP_OK));
        }

    private:
        TCaptchaFallbackStateAggregator& FallbackStateAggregator;
        bool Activate;
    };

    class THandlerStat: public IHandler {
    public:
        THandlerStat(TCaptchaStats& stats)
            : Stats(stats)
        {
        }

        NThreading::TFuture<THttpResponse> HandleRequest(TRequestInfo&) override {
            THttpResponse response;
            response.SetContent(Stats.CreateJsonDump());
            return NThreading::MakeFuture(response);
        }

    private:
        TCaptchaStats& Stats;
    };

    class THandlerReopenLogs: public IHandler {
    public:
        THandlerReopenLogs(TCaptchaHttpServer* server)
            : Server(server)
        {
        }

        NThreading::TFuture<THttpResponse> HandleRequest(TRequestInfo& reqinfo) override {
            if (!reqinfo.Local) {
                THttpResponse response;
                response.SetHttpCode(HTTP_FORBIDDEN);
                return NThreading::MakeFuture(response);
            }

            Server->ReopenLogs();
            return NThreading::MakeFuture(THttpResponse());
        }

    private:
        TCaptchaHttpServer* Server;
    };

    void TCaptchaHttpServer::ReopenLogs() {
        TLoggerOperator<TGlobalLog>::Log().ReopenLog();
        ChecksLog.ReopenLog();
        GenerateLog.ReopenLog();
        if (AccessLog) {
            AccessLog->ReopenLog();
        }
    }

    static THttpServer::TOptions MakeOptions(const TCaptchaConfig& config) {
        THttpServer::TOptions result;

        result.Port = config.GetHttpServer().GetPort();
        result.nThreads = config.GetHttpServer().GetNumThreads();
        result.MaxQueueSize = config.GetHttpServer().GetMaxQueueSize();
        result.MaxConnections = config.GetHttpServer().GetMaxConnections();
        result.ListenBacklog = config.GetHttpServer().GetListenBacklog();
        result.KeepAliveEnabled = config.GetHttpServer().GetKeepAliveEnabled();

        return result;
    }

    TCaptchaHttpServer::TCaptchaHttpServer(const TCaptchaConfig& config, TCaptchaStats& stats, TCaptchaSessionFactory& sessionFactory, TCaptchaItemsStorageRouter& itemsStorageRouter, TCaptchaSessionStorageRouter& sessionStorageRouter, TCaptchaFallbackStateAggregator& fallbackStateAggregator)
        : THttpServer(this, MakeOptions(config))
        , Stats(stats)
        , SessionFactory(sessionFactory)
        , Config(config)
        , ItemsStorageRouter(itemsStorageRouter)
        , SessionStorageRouter(sessionStorageRouter)
        , FallbackStateAggregator(fallbackStateAggregator)
        , RequestRateMonitor(config, fallbackStateAggregator)
        , ChecksLog(MakeHolder<TThreadedLogBackend>(new TFileLogBackend(Config.GetChecksLogPath()), Config.GetChecksLogQueueLength()))
        , GenerateLog(MakeHolder<TThreadedLogBackend>(new TFileLogBackend(Config.GetGenerateLogPath()), Config.GetGenerateLogQueueLength()))
    {
        Handlers["/ping.html"] = MakeHolder<THandlerPing>(Config);

        Handlers["/debug/items_index"] = MakeHolder<THandlerItemsIndex>(ItemsStorageRouter);
        Handlers["/debug/preload_items"] = MakeHolder<THandlerPreloadItems>(ItemsStorageRouter);
        Handlers["/debug/fallback_list"] = MakeHolder<THandlerFallbackList>(SessionStorageRouter);

        Handlers["/admin/fallback_activate"] = MakeHolder<THandlerSetFallbackState>(FallbackStateAggregator, true);
        Handlers["/admin/fallback_deactivate"] = MakeHolder<THandlerSetFallbackState>(FallbackStateAggregator, false);

        Handlers["/stat"] = MakeHolder<THandlerStat>(Stats);
        Handlers["/reopen_logs"] = MakeHolder<THandlerReopenLogs>(this);

        Handlers["/generate"] = MakeHolder<THandlerGenerate>(this, Config, SessionStorageRouter, GenerateLog);
        Handlers["/image"] = MakeHolder<THandlerImage>(this, SessionStorageRouter);
        Handlers["/voice"] = MakeHolder<THandlerVoice>(this, ItemsStorageRouter, SessionStorageRouter);
        Handlers["/check"] = MakeHolder<THandlerCheck>(Config, this, SessionStorageRouter, ChecksLog);
        Handlers["/answer"] = MakeHolder<THandlerAnswer>(Config, this, SessionStorageRouter);

        Handlers["/static"] = MakeHolder<THandlerStatic>(Config);

        if (Config.GetEnableTestingHandlers()) {
            Handlers["/randomitem"] = MakeHolder<THandlerRandomItem>(ItemsStorageRouter);
        }

        Stats.RegisterSignalCallback(ESignals::RequestQueueSize, [this]() { return double(GetRequestQueueSize()); });
        Stats.RegisterSignalCallback(ESignals::ConnectionCount, [this]() { return double(GetClientCount()); });

        TDeque<TString> handlerNames;
        for (const auto& handler : Handlers) {
            handlerNames.push_back(handler.first);
        }
        Stats.InitHandlerSegmentedSignals(handlerNames);

        const auto& accessLogPath = config.GetAccessLogPath();
        if (accessLogPath) {
            AccessLog = MakeHolder<TLog>(MakeHolder<TThreadedLogBackend>(new NSnooper::TMaskedLogBackend(new TFileLogBackend(accessLogPath)), Config.GetAccessLogQueueLength()));
        }
    }
}
