#pragma once

#include "random.h"

#include <library/cpp/json/json_value.h>
#include <library/cpp/threading/future/future.h>

#include <util/generic/hash.h>
#include <util/generic/deque.h>
#include <util/generic/string.h>

#include <tuple>

namespace NCaptchaServer {
    struct TCaptchaItemKey {
        TString Version;
        TString Type;
        TString Id;

        TCaptchaItemKey() {
        }
        TCaptchaItemKey(const TString& version, const TString& type, const TString& id)
            : Version(version)
            , Type(type)
            , Id(id)
        {
        }
        TCaptchaItemKey(const NJson::TJsonValue& jsonVal);

        NJson::TJsonValue AsJsonValue() const;

        bool operator==(const TCaptchaItemKey& rhs) const {
            return Version == rhs.Version && Type == rhs.Type && Id == rhs.Id;
        }

        TString DebugStr() const {
            return Version + ";" + Type + ";" + Id;
        }
    };

    struct TCaptchaItemHeader {
        TCaptchaItemKey Key;
        NJson::TJsonValue Metadata;

        TCaptchaItemHeader() {
        }
        TCaptchaItemHeader(const TCaptchaItemKey& key, const NJson::TJsonValue& metadata)
            : Key(key)
            , Metadata(metadata)
        {
        }
    };

    using TCaptchaItemsIndex = TDeque<TCaptchaItemHeader>;

    class ICaptchaItemsStorage {
    public:
        virtual NThreading::TFuture<bool> LoadItemData(const TCaptchaItemKey& key, TString& data) = 0;
        virtual void LoadIndex(TCaptchaItemsIndex& index) = 0;
        virtual void LoadIndex(TStringBuf type, TCaptchaItemsIndex& index);
        virtual TCaptchaItemHeader LoadRandomItemHeader(TStringBuf type, IRng* rng);
        TCaptchaItemHeader LoadRandomItemHeader(TStringBuf type) {
            TDefaultRng rng;
            return LoadRandomItemHeader(type, &rng);
        }
        virtual ~ICaptchaItemsStorage() {
        }
    };
}

template <>
struct THash<NCaptchaServer::TCaptchaItemKey> {
    inline size_t operator()(const NCaptchaServer::TCaptchaItemKey& key) const {
        auto tup = std::tie(key.Version, key.Type, key.Id);
        return THash<decltype(tup)>()(tup);
    }
};
