#include "localization_checks.h"

#include <util/generic/xrange.h>
#include <util/string/split.h>

namespace NCaptchaServer {
    TString TRectangle::AsFlatString() const {
        TString result;
        TStringOutput so(result);
        so << FloatToString(MinX)
           << "," << FloatToString(MinY)
           << ";" << FloatToString(MaxX)
           << "," << FloatToString(MaxY);
        return result;
    }

    TRectangle TRectangle::FromFlatString(const TString& str) {
        TVector<TString> coords;
        StringSplitter(str).SplitBySet(",;").AddTo(&coords);

        TRectangle result;
        result.MinX = FromString<double>(coords[0]);
        result.MinY = FromString<double>(coords[1]);
        result.MaxX = FromString<double>(coords[2]);
        result.MaxY = FromString<double>(coords[3]);
        return result;
    }

    static TRectangle Intersect(const TRectangle& rect1, const TRectangle& rect2) {
        TRectangle result;
        result.MinX = Max(rect1.MinX, rect2.MinX);
        result.MinY = Max(rect1.MinY, rect2.MinY);
        result.MaxX = Min(rect1.MaxX, rect2.MaxX);
        result.MaxY = Min(rect1.MaxY, rect2.MaxY);
        return result;
    }

    static double Area(const TRectangle& rect) {
        if (rect.MaxX < rect.MinX || rect.MaxY < rect.MinY)
            return .0;

        return (rect.MaxX - rect.MinX) * (rect.MaxY - rect.MinY);
    }

    TLocalizationChecker::TLocalizationChecker(const TRectangle& bbox, const TRectangle& answer, const TResponseArray& reply, const TGrid& grid)
        : Reply(reply)
        , GridBboxes(Reserve(grid.X * grid.Y))
    {
        GridXStep = (bbox.MaxX - bbox.MinX) / grid.X;
        GridYStep = (bbox.MaxY - bbox.MinY) / grid.Y;

        for (auto y : xrange(grid.Y)) {
            for (auto x : xrange(grid.X)) {
                TRectangle b;
                b.MinX = bbox.MinX + x * GridXStep;
                b.MinY = bbox.MinY + y * GridYStep;
                b.MaxX = bbox.MinX + (x + 1) * GridXStep;
                b.MaxY = bbox.MinY + (y + 1) * GridYStep;
                GridBboxes.push_back(b);
            }
        }

        BboxAnswerCropped = Intersect(bbox, answer);
    }

    double TLocalizationChecker::IntersectOverUnion() const {
        double answerSize = Area(BboxAnswerCropped);
        double replySize = GridXStep * GridYStep * Reply.size();

        double intersectSize = .0;
        for (auto r : Reply) {
            const auto& replyBbox = GridBboxes.at(r);
            TRectangle intersectBbox = Intersect(replyBbox, BboxAnswerCropped);
            intersectSize += Area(intersectBbox);
        }

        return intersectSize / (answerSize + replySize - intersectSize);
    }

    bool TLocalizationChecker::AllUserCellsIntersectWithAnswer() const {
        for (auto r : Reply) {
            const auto& replyBbox = GridBboxes.at(r);
            if (Area(Intersect(replyBbox, BboxAnswerCropped)) == .0) {
                return false;
            }
        }
        return true;
    }
}
