#include "mode_file_watcher.h"

#include <util/stream/file.h>
#include <util/string/strip.h>

namespace NCaptchaServer {
    TCaptchaModeFileWatcher::TCaptchaModeFileWatcher(const TCaptchaConfig& config, TCaptchaFallbackStateAggregator& fallbackStateAggregator)
        : FallbackStateAggregator(fallbackStateAggregator)
        , PollerThread(PollLoop, this)
    {
        const auto& options = config.GetModeFileWatcher();

        Path = options.GetPath();
        PollInterval = TDuration::MilliSeconds(options.GetPollIntervalMs());

        for (const auto& kw : options.GetEnableFallbackKeywords()) {
            EnableFallbackKeywords.insert(kw);
        }

        for (const auto& kw : options.GetDisableFallbackKeywords()) {
            DisableFallbackKeywords.insert(kw);
        }

        ReadFile();
        PollerThread.Start();
    }

    TCaptchaModeFileWatcher::~TCaptchaModeFileWatcher() {
        StopEvent.Signal();
        PollerThread.Join();
    }

    void* TCaptchaModeFileWatcher::PollLoop(void* ptr) {
        TCaptchaModeFileWatcher* thisptr = reinterpret_cast<TCaptchaModeFileWatcher*>(ptr);
        while (true) {
            if (thisptr->StopEvent.WaitT(thisptr->PollInterval)) {
                return nullptr;
            }

            try {
                thisptr->ReadFile();
            } catch (std::exception& ex) {
                ERROR_LOG << "Error in mode file poll loop: " << ex.what() << Endl;
            } catch (...) {
                ERROR_LOG << "Error in mode file poll loop (unknown type)" << Endl;
            }
        }
    }

    void TCaptchaModeFileWatcher::ReadFile() {
        TString mode;
        try {
            mode = Strip(TFileInput(Path).ReadAll());
        } catch (TFileError&) {
            mode = "";
        }
        bool fallbackState;

        if (EnableFallbackKeywords.contains(mode)) {
            fallbackState = true;
        } else if (DisableFallbackKeywords.contains(mode)) {
            fallbackState = false;
        } else {
            ythrow yexception() << "Unknown mode: " << mode.Quote();
        }

        if (LastFallbackState != fallbackState) {
            TStringBuf action = fallbackState ? TStringBuf("Activating") : TStringBuf("Deactivating");
            NOTICE_LOG << action << " fallback based on mode file value " << mode.Quote() << Endl;
            FallbackStateAggregator.SetFallbackState(ECaptchaFallbackActivator::ModeFileWatcher, fallbackState);
            LastFallbackState = fallbackState;
        }
    }
}
