#include "precaching_session_storage.h"

#include <library/cpp/logger/global/global.h>
#include <util/generic/utility.h>

namespace NCaptchaServer {
    TCaptchaPrecachingSessionStorage::TSessionCache::TSessionCache(TCaptchaStats& stats, TMutex& cacheItemsMutex, TCondVar& cacheItemDropCondVar, const TCaptchaPrecachingSessionStorage::TCacheKey& key, ui32 maxSessions)
        : Stats(stats)
        , CacheItemsMutex(cacheItemsMutex)
        , CacheItemDropCondVar(cacheItemDropCondVar)
        , CaptchaType(std::get<0>(key))
        , VoiceType(std::get<1>(key))
        , Checks(std::get<2>(key))
        , MaxSessions(maxSessions)
    {
    }

    bool TCaptchaPrecachingSessionStorage::TSessionCache::IsFull() const {
        auto guard = Guard(CacheItemsMutex);

        return Cache.size() >= MaxSessions;
    }

    ui32 TCaptchaPrecachingSessionStorage::TSessionCache::Size() const {
        auto guard = Guard(CacheItemsMutex);

        return Cache.size();
    }

    ui32 TCaptchaPrecachingSessionStorage::TSessionCache::UnfilledCount() const {
        auto guard = Guard(CacheItemsMutex);

        if (Cache.size() >= MaxSessions) {
            return 0;
        } else {
            return MaxSessions - Cache.size();
        }
    }

    TInstant TCaptchaPrecachingSessionStorage::TSessionCache::OldestTimestamp() const {
        auto guard = Guard(CacheItemsMutex);

        if (!Cache.empty()) {
            return Cache.front().SessionInfo.Timestamp;
        } else {
            return TInstant::Max();
        }
    }

    void TCaptchaPrecachingSessionStorage::TSessionCache::AddSession(TCachedSessionInfo&& session) {
        auto guard = Guard(CacheItemsMutex);

        Cache.push_back(session);
    }

    bool TCaptchaPrecachingSessionStorage::TSessionCache::GetSession(TInstant notBefore, TCachedSessionInfo& session) {
        auto guard = Guard(CacheItemsMutex);
        DoDropExpiredSessions(notBefore);

        if (!Cache.empty()) {
            session = std::move(Cache.front());
            Cache.pop_front();
            CacheItemDropCondVar.BroadCast();
            return true;
        }
        return false;
    }

    void TCaptchaPrecachingSessionStorage::TSessionCache::DropExpiredSessions(TInstant notBefore) {
        auto guard = Guard(CacheItemsMutex);

        DoDropExpiredSessions(notBefore);
    }

    void TCaptchaPrecachingSessionStorage::TSessionCache::DoDropExpiredSessions(TInstant notBefore) {
        while (!Cache.empty() && Cache.front().SessionInfo.Timestamp < notBefore) {
            Stats.PushSignal(ESessionCacheKeySegmentedSignals::SessionPrecachingExpired, CaptchaType, VoiceType, Checks);
            Stats.PushSignal(ESignals::SessionPrecachingExpired);
            Cache.pop_front();
            CacheItemDropCondVar.BroadCast();
        }
    }

    TCaptchaPrecachingSessionStorage::TCaptchaPrecachingSessionStorage(const TCaptchaConfig& config, TCaptchaStats& stats, ICaptchaSessionStorage* slave)
        : ProviderGuard(AvailabilityMonitor)
        , Config(config)
        , Stats(stats)
        , Slave(slave)
        , MaxCacheSize(0)
        , RefillerThread(RefillLoop, this)
    {
        FillTypeAliases(config.GetCaptchaTypes(), CaptchaTypeAliases);
        FillTypeAliases(config.GetVoiceCaptchaTypes(), VoiceCaptchaTypeAliases);

        SessionReservedTime = TDuration::Seconds(Config.GetSessionCache().GetSessionReservedTimeSeconds());
        MaxAwaitingResponse = Config.GetSessionCache().GetMaxAwaitingResponse();

        for (auto bucket : Config.GetSessionCache().GetBuckets()) {
            const auto& key = bucket.GetKey();
            const TString& ctype = GetWithDefault(CaptchaTypeAliases, key.GetType());
            const TString& vtype = GetWithDefault(VoiceCaptchaTypeAliases, key.GetVoiceType());
            AddCache(std::make_tuple(ctype, vtype, key.GetChecks()), bucket.GetMaxSessions());
            MaxCacheSize += bucket.GetMaxSessions();
        }

        Stats.RegisterSignalCallback(ESignals::SessionPrecachingAwaitingResponse, [this]() { with_lock (AwaitingResponseMutex) { return AwaitingResponse; } });

        Stats.RegisterSignalCallback(ESignals::SessionPrecachingCacheSize, [this]() {
            return double(TotalCacheSize());
        });

        Stats.RegisterSignalCallback(ESignals::SessionPrecachingTimeSinceOldestSessionMs, [this]() {
            TInstant oldest = TInstant::Max();
            for (auto& iter : Caches) {
                oldest = Min(oldest, iter.second.OldestTimestamp() - SessionReservedTime);
            }

            if (oldest != TInstant::Max()) {
                return double((Now() - oldest).MilliSeconds());
            } else {
                return .0;
            }
        });

        Stats.RegisterSignalCallback(ESessionCacheKeySegmentedSignals::SessionPrecachingTimeSinceOldestSessionMs, [this](const TString& ctype, const TString& vtype, ui32 checks) {
            const auto& cache = Caches.at(std::make_tuple(ctype, vtype, checks));
            TInstant oldest = cache.OldestTimestamp() - SessionReservedTime;

            if (oldest != TInstant::Max()) {
                return double((Now() - oldest).MilliSeconds());
            } else {
                return .0;
            }
        });

        AtomicSet(Exiting, 0);

        RefillerThread.Start();
    }

    char TCaptchaPrecachingSessionStorage::TokenTag() {
        return Slave->TokenTag();
    }

    NThreading::TFuture<TString> TCaptchaPrecachingSessionStorage::CreateSession(const TCaptchaSessionRequest& request, TCaptchaSessionInfo& info) {
        TCachedSessionInfo cachedSession;

        const TString& ctype = GetWithDefault(CaptchaTypeAliases, request.Type);
        const TString& vtype = GetWithDefault(VoiceCaptchaTypeAliases, request.VoiceType);
        ui32 checks = request.Checks > 0 ? request.Checks : 0;

        auto key = std::make_tuple(ctype, vtype, checks);
        auto cache = Caches.find(key);
        if (cache != Caches.end()) {
            if (cache->second.GetSession(request.Timestamp, cachedSession)) {
                Stats.PushSignal(ESignals::SessionPrecachingHits);
                info = cachedSession.SessionInfo;
                return NThreading::MakeFuture(cachedSession.Token);
            } else {
                Stats.PushSignal(ESignals::SessionPrecachingDefinedMisses);
            }
        }

        Stats.PushSignal(ESignals::SessionPrecachingMisses);
        return Slave->CreateSession(request, info);
    }

    NThreading::TFuture<bool> TCaptchaPrecachingSessionStorage::LoadSessionInfo(TStringBuf token, TCaptchaSessionInfo& info) {
        return Slave->LoadSessionInfo(token, info);
    }

    NThreading::TFuture<bool> TCaptchaPrecachingSessionStorage::StoreSessionInfo(TStringBuf token, const TCaptchaSessionInfo& info) {
        return Slave->StoreSessionInfo(token, info);
    }

    NThreading::TFuture<void> TCaptchaPrecachingSessionStorage::DropSession(TStringBuf token) {
        return Slave->DropSession(token);
    }

    TCaptchaPrecachingSessionStorage::~TCaptchaPrecachingSessionStorage() {
        AtomicSet(Exiting, 1);
        CacheItemDropCondVar.BroadCast();
        ReceivedResponseCondVar.BroadCast();
        RefillerThread.Join();
    }

    void TCaptchaPrecachingSessionStorage::AddCache(const TCaptchaPrecachingSessionStorage::TCacheKey& key, ui32 maxSessions) {
        Caches.emplace(std::piecewise_construct, std::forward_as_tuple(key), std::forward_as_tuple(Stats, CacheItemsMutex, CacheItemDropCondVar, key, maxSessions));
        CacheKeys.push_back(key);
    }

    void TCaptchaPrecachingSessionStorage::DropExpiredSessions() {
        for (auto& iter : Caches) {
            iter.second.DropExpiredSessions(Now());
        }
    }

    void TCaptchaPrecachingSessionStorage::PrepareNewSession(const TCaptchaPrecachingSessionStorage::TCacheKey& key) {
        TInstant timestamp = Now() + SessionReservedTime;
        TCaptchaSessionRequest sessreq(std::get<0>(key), std::get<1>(key), std::get<2>(key), timestamp);
        TAtomicSharedPtr<TCaptchaSessionInfo> sessionInfoPtr(new TCaptchaSessionInfo);

        auto futureSession = Slave->CreateSession(sessreq, *sessionInfoPtr);
        with_lock (AwaitingResponseMutex) {
            AwaitingResponse++;
        }

        TResourceAvailabilityMonitor mon(AvailabilityMonitor);
        futureSession.Subscribe([this, key, sessionInfoPtr, mon](const NThreading::TFuture<TString>& ftoken) {
            TResourceAccessorGuard ag(mon);
            if (!ag) {
                // Response to CreateSession arrived after this object was destroyed. Nothing to do in that case.
                return;
            }

            with_lock (AwaitingResponseMutex) {
                AwaitingResponse--;
                ReceivedResponseCondVar.BroadCast();
            }
            try {
                TCachedSessionInfo info(ftoken.GetValue(), std::move(*sessionInfoPtr));
                Caches.at(key).AddSession(std::move(info));
            } catch (...) {
                ERROR_LOG << "Exception while preparing a cached session: " << CurrentExceptionMessage() << Endl;
                Stats.PushSignal(ESignals::TotalExceptions);
            }
        });
    }

    ui32 TCaptchaPrecachingSessionStorage::LoadNewSessions(ui32 count) {
        auto size = Caches.size();
        TVector<ui32> freeSlots(Reserve(Caches.size()));
        ui32 totalRemaining = 0;
        for (auto i : xrange(Caches.size())) {
            const auto& key = CacheKeys[i];
            ui32 cur = Caches.at(key).UnfilledCount();

            freeSlots.push_back(cur);
            totalRemaining += cur;
        }

        if (count > totalRemaining) {
            count = totalRemaining;
        }

        ui32 newSessions = 0;
        ui32 filledStreak = 0;
        while (newSessions < count) {
            auto index = CachesRoundRobin;
            CachesRoundRobin = (CachesRoundRobin + 1) % size;

            if (!freeSlots[index]) {
                filledStreak += 1;
                Y_ASSERT(filledStreak <= size);
                continue;
            }

            PrepareNewSession(CacheKeys[index]);

            newSessions += 1;
            freeSlots[index] -= 1;
            filledStreak = 0;
        }

        return newSessions;
    }

    ui32 TCaptchaPrecachingSessionStorage::TotalCacheSize() const {
        ui32 result = 0;
        for (auto& iter : Caches) {
            result += iter.second.Size();
        }
        return result;
    }

    TInstant TCaptchaPrecachingSessionStorage::NextExpiration() const {
        TInstant result = TInstant::Max();
        for (auto& iter : Caches) {
            result = Min(result, iter.second.OldestTimestamp());
        }

        return result;
    }

    void* TCaptchaPrecachingSessionStorage::RefillLoop(void* ptr) {
        TCaptchaPrecachingSessionStorage* thisptr = reinterpret_cast<TCaptchaPrecachingSessionStorage*>(ptr);

        while (true) {
            try {
                thisptr->DropExpiredSessions();

                ui32 freeSlots = 0;
                with_lock (thisptr->CacheItemsMutex) {
                    thisptr->CacheItemDropCondVar.WaitD(thisptr->CacheItemsMutex, thisptr->NextExpiration(), [thisptr]() {
                        return thisptr->TotalCacheSize() < thisptr->MaxCacheSize || AtomicGet(thisptr->Exiting);
                    });

                    freeSlots = thisptr->MaxCacheSize - thisptr->TotalCacheSize();
                }

                if (AtomicGet(thisptr->Exiting)) {
                    return nullptr;
                }

                ui32 allowedRequests = 0;
                with_lock (thisptr->AwaitingResponseMutex) {
                    thisptr->ReceivedResponseCondVar.WaitI(thisptr->AwaitingResponseMutex, [thisptr]() {
                        return thisptr->AwaitingResponse < thisptr->MaxAwaitingResponse || AtomicGet(thisptr->Exiting);
                    });

                    allowedRequests = thisptr->MaxAwaitingResponse - thisptr->AwaitingResponse;
                }

                if (AtomicGet(thisptr->Exiting)) {
                    return nullptr;
                }

                thisptr->LoadNewSessions(Min(allowedRequests, freeSlots));
            } catch (std::exception& ex) {
                ERROR_LOG << "Error in session cache update loop: " << ex.what() << Endl;
            } catch (...) {
                ERROR_LOG << "Error in session cache update loop (unknown type)" << Endl;
            }
        }
    }
}
